/* $Id: ArkPrimBlock.h,v 1.13 2003/03/26 15:43:01 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_PRIMBLOCK_H
#define ARK_PRIMBLOCK_H

#include <Ark/ArkTypes.h>
#include <Ark/ArkMath.h>
#include <vector>
#include <list>

namespace Ark
{
   enum PrimitiveType
   {
      PRIM_NULL = 0,
      PRIM_TRIANGLES,
      PRIM_TRIANGLE_FAN,
      PRIM_TRIANGLE_STRIP
   };

   /**
    * A primitive block is simply a list of primitives sharing the same type
    * and material. It contains an array of vertex indices ; those indices
    * refers to the vertex buffer this primitive block is associated with.
    */
   class ARK_DLL_API PrimitiveBlock
   {
         std::vector< ushort > m_List;
	 PrimitiveType m_Type;
	 size_t m_EnabledSize;

      public:
	 /** Create an empty primitive block */
	 inline PrimitiveBlock()
	 {
	    m_Type = PRIM_NULL;
	 }

	 /** Destroy the primitive block. */
	 ~PrimitiveBlock() {}

	 /**
	  * This function sets the type of the primitive block. It does not
	  * reset indice data, but you'll probably have to change them by hand
	  * since changing the types usually means changing the shape of the 
	  * object.
	  */
	 inline void SetType (PrimitiveType type)
	 {m_Type = type;}
	 
	 /** Return the type of the primitive block. */
	 inline PrimitiveType Type() const
	 {return m_Type;}

	 /**
	  * Changes the number of enabled primitive' indices. By default, the
	  * number of enabled indices is set to the number of indices in
	  * the primitive block. Using this function you might have a static
	  * primitive block with \c Size() indices and drawing only
	  * \c EnabledSize() indices.
	  */
	 inline void SetEnabledSize(size_t size)
	 {
	    assert (size <= Size());
	    m_EnabledSize = size;
	 }

	 /// Return the number of actually used indices.
	 inline size_t EnabledSize() const {return m_EnabledSize;}

	 /// Return the space used by this block.
	 inline size_t Size() const {return m_List.size();}

	 /// Reserve some space for triangle indices...
	 inline void Resize (size_t size) { m_List.resize(size);}

	 /// Add an index to the list
	 inline void Add (ushort index)
	 { m_List.push_back( index ); }

	 /// Pass the operator [] to the list
	 const ushort& operator[](size_t index) const 
	 { return m_List[index]; }

	 /// Pass the operator [] to the list
	 ushort& operator[](size_t index)
	 { return m_List[index]; }
   };

   // Useful typedefs
   typedef std::list< PrimitiveBlock > BlockList;
   typedef BlockList::iterator BlockLI;
   typedef BlockList::const_iterator BlockLCI;

};

#endif
