/* $Id: ArkParticle.cpp,v 1.14 2002/10/11 01:10:04 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkParticle.h>
#include <Ark/ArkWorld.h>
#include <Ark/ArkRenderer.h>

#include <stdlib.h>
#include <stdio.h>

namespace Ark
{
   
   /* ====================== PARTICLES ============ */
   scalar frand (float max = 1.0f)
   {
      return max * ((rand ()/scalar(RAND_MAX) * 2.0f) - 1.0f);
   }
   
   scalar frand (float min, float max)
   {
      return min + (rand ()/scalar(RAND_MAX) * (max-min)) ;
   }
   
   Particle::Particle (ParticleSys *sys)
   {
   }
   
   Particle::~Particle ()
   {
   }
   
   
   void
   Particle::Init (ParticleSys *system)
   {
      ParticleTmpl *tpl = system->m_Template;
      assert (tpl);
      
      if (tpl->m_Flags & ParticleTmpl::DIE_AGE)
      {
	 m_LifeTime = frand (tpl->m_MinLife,
			     tpl->m_MaxLife);
      }
      
      system->GetNew (&m_Pos, &m_Vel);
      
      m_Age = 0.0;
      m_Dead = false;
   }
   
   void
   Particle::Update (ParticleSys *system, scalar dt)
   {
      ParticleTmpl *tpl = system->m_Template;
      assert (tpl);
      
      m_Age += dt;
      
      if ((tpl->m_Flags & ParticleTmpl::DIE_AGE) > 0
	  && m_Age > m_LifeTime)
      {
	 m_Dead = true;
	 return;
      }
      
      // DO PARTICLE COLLISION
      Vector3 add (tpl->m_Radius, tpl->m_Radius,
		   tpl->m_Radius);
      BBox bb;
      
      bb.m_Min = m_Pos + add;
      bb.m_Max = m_Pos - add;
      
      if ((tpl->m_Flags & ParticleTmpl::DIE_COLLISION) > 0
	  && system->m_World)
      {
	  std::vector< Collision > cols;
	 
	 system->m_World->TestCollision (bb, Collision::ONESHOT, cols);
	 
	 if (cols.size())
	    m_Dead = true;
      }
      
      /* Compute the acceleration */
      Vector3 v(0.0,0.0,0.0);
      
      if (tpl->m_Flags & ParticleTmpl::FCT_GRAVITY)
	 v += Vector3 (system->m_Gravity).Scale (tpl->m_Weigth);
      
      if (tpl->m_Flags & ParticleTmpl::FCT_WIND)
      {
	 // The more the object is heavy, the less the wind can
	 // move it. (hum hum.. I dont have any physic knowledge at the
	 // time I write this..)
	 v += Vector3 (system->m_Wind).Scale (1.0f / tpl->m_Weigth);
      }
      
      if (tpl->m_Flags & ParticleTmpl::FCT_RANDOM)
      {
	 v += Vector3 (frand(), frand(), frand()).Scale
	    (tpl->m_RandomScale);
      }
      
      if (tpl->m_Flags & ParticleTmpl::FCT_STRENGTH)
	 v += tpl->m_Strength;
      
      m_Vel += v.Scale (dt);
      m_Pos += Vector3(m_Vel).Scale (dt);
   }
   
   /* ====================== PARTICLE SYSTEMS ============ */
   
   ParticleSys::ParticleSys (World *world, Vector3 pos,
			     Vector3 initvel, scalar radius) : 
      m_World (world),
      m_Stopped (false),
      m_Template (NULL),
      m_Pos (pos),
      m_Velocity (initvel),
      m_Gravity (0.0f, -0.98f, 0.0f),
      m_Radius (radius)
   {
      SetDefaultPos (D_RANDOM, D_RANDOM, D_RANDOM);
      SetDefaultVel (D_GIVEN, D_GIVEN, D_GIVEN);
   }
   
   static ParticleSys::Default
   GetDefault (char c)
   {
      if (c == 'R')
	 return ParticleSys::D_RANDOM;
      else if (c == 'G')
	 return ParticleSys::D_GIVEN;
      else
	 return ParticleSys::D_NULL;
   }
   
   ParticleSys::ParticleSys (World *world,
			     const String &posvel,
			     const String &defs,
			     const String &part,
			     Cache *cache) : 
      m_World (NULL),
      m_Stopped (false),
      m_Template (NULL),
      m_Gravity (0.0f, -0.98f, 0.0f)
   {
      sscanf (posvel.c_str(), "(%f,%f,%f)(%f,%f,%f)(%f)",
	      &m_Pos.X, &m_Pos.Y, &m_Pos.Z,
	      &m_Velocity.X, &m_Velocity.Y, &m_Velocity.Z,
	      &m_Radius);
      
      if (defs.size() >= 6)
      {
	 char sdefs[6];
	 
	 sscanf (defs.c_str(), "(%c, %c, %c)(%c, %c, %c)",
		 &sdefs[0], &sdefs[1], &sdefs[2],
		 &sdefs[3], &sdefs[4], &sdefs[5]);
	 
	 SetDefaultPos (GetDefault (sdefs[0]),
			GetDefault (sdefs[1]),
			GetDefault (sdefs[2]));
	 
	 SetDefaultVel (GetDefault (sdefs[3]),
			GetDefault (sdefs[4]),
			GetDefault (sdefs[5]));
      }
      else
      {
	 SetDefaultPos (D_RANDOM, D_RANDOM, D_RANDOM);
	 SetDefaultVel (D_GIVEN, D_GIVEN, D_GIVEN);
      }
      
      int nparts;
      char name[150];
      
      if (sscanf (part.c_str(), "(%d,%150s)", &nparts, name) == 2)
      {
	 ParticleTmpl *tpl;
	 cache->Get (V_PARTICLE, name, &tpl);
	 Insert (tpl, nparts);
	 
	 tpl->Unref();
      }
   }
   
   ParticleSys::~ParticleSys ()
   {
      if (m_Template)
	 m_Template->Unref();
      Clear ();
   }
   
   void
   ParticleSys::Clear ()
   {
      m_Particles.clear();
   }
   
   void
   ParticleSys::GetNew (Vector3 *pos, Vector3 *vel)
   {
      assert (pos);
      assert (vel);
      
      /* Compute position */
      Default *def = m_DefPos;
      scalar *real = &pos->X;
      int i;
      
      for (i = 0; i < 3; i++)
      {
	 switch (def [i])
	 {
	    case D_NULL:    real[i] = 0.0; break;
	    case D_GIVEN:   real[i] = (&m_Pos.X)[i]; break;
	    case D_RANDOM:  real[i] = (&m_Pos.X)[i] + frand() * m_Radius;
	       break;
	 }
      }
      
      /* Compute velocity */
      def = m_DefVel;
      real = &vel->X;
      
      for (i = 0; i < 3; i++)
      {
	 switch (def [i])
	 {
	    case D_NULL:    real	[i] = 0.0; break;
	    case D_GIVEN:   real[i] = (&m_Velocity.X)[i]; break;
	    case D_RANDOM:  real[i] = frand(); break;
	 }
      }
   }
   
   void
   ParticleSys::Insert (ParticleTmpl *tpl, int n)
   {
      if (tpl == NULL)
	 return;

      if (m_Template) 
	 m_Template->Unref();

      m_Template = tpl;
      m_Template->Ref();
      
      m_Particles.insert (m_Particles.begin(), n, Particle (this));
      for (int i = 0; i < n; i++)
	 m_Particles[i].Init (this);
   }
   
   void
   ParticleSys::Update (scalar dt)
   {
      if (m_Particles.empty() || m_Template == NULL)
	 return;
      
      std::vector<Particle>::iterator i;
      
      // We do not increment iterator in 'for' construct because erasing
      //  elements will invalidate iterator
      for (i = m_Particles.begin(); i != m_Particles.end() ;)
      {
	  bool erase_current = false;

	  if (i->m_Dead == false)
	  {
	      // FIXME : Is this can change i->m_dead ? if so, should be documented
	      i->Update (this, dt);
	  }

	  if (i->m_Dead)
	  {
	      if (m_Template->m_Flags & ParticleTmpl::DIE_RESPAWN
		      && !m_Stopped)
		  i->Init (this);
	      else
	      {
		  erase_current = true;
	      }
	  }

	  // Do correct erasing and valid iterators
	  if ( erase_current )
	  {
	      // if we remove, gets the iteratoe from the erase function
	      i = m_Particles.erase (i);
	  }
	  else
	  {
	      // we do not remove, increment operator
	      ++i;
	  }
      }
   }

   void
   ParticleSys::Render (Renderer &renderer)
   {
      renderer.RenderParticleSys (this);
   }
   
   /* ============================= TEMPLATES ====== */
   
   ParticleTmpl::ParticleTmpl (const String &name) :
      Object (name)
   {
      m_pType = TYPE_NULL;
      m_Type = V_PARTICLE;
      m_Radius = 0.1f;
      m_RandomScale = 1.0f;
   }
   
   ParticleTmpl::~ParticleTmpl()
   {
      if (m_pType == TYPE_SPRITE && m_Sprite)
	 m_Sprite->Unref();
   }
   
   void
   ParticleTmpl::Write (WriteStream &stream)
   {
   }

}
