/* $$
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <Modules/Lua/LuaTable.h>
#include <iostream>

namespace Ark
{

/******************************************************************************
 * Construct a LuaTable, the table being at the top of the stack
 ******************************************************************************/
LuaTable::LuaTable(lua_State* L, int index)
  : m_L(L), m_Index(index)
{
  if(m_Index == -1)
  {
    m_Index = lua_gettop(m_L);
  }
  else if(m_Index < -1)
  {
    std::cerr
      << "Warning: A LuaTable has a relative stack index, "
      << "which may cause some bugs..."
      << std::endl;
  }
}

/******************************************************************************
 * Destructor
 ******************************************************************************/
LuaTable::~LuaTable()
{
  lua_pop(m_L, m_Index);
}

/******************************************************************************
 * Return the table's index on the stack
 ******************************************************************************/
int LuaTable::index() const
{
  return m_Index;
}

/******************************************************************************
 * The following functions get a value of the table on the stack, given
 * the key of the value in the table.
 *
 * TODO: add the CFunction type.
 ******************************************************************************/

/******************************************************************************
 * Read a number in the table.
 * 
 * \param key The key of the element to get in the table.
 * \return The value at the given key or 0 if the value is not a number
 * or is not a string convertible to a number.
 ******************************************************************************/
double LuaTable::getNumber(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);
  
  int result = static_cast<int>(lua_tonumber(m_L, -1));
  lua_pop(m_L, 1);

  return result;
}

/******************************************************************************
 * Read a string in the table.
 *
 * \param key The key of the element to get in the table.
 * \return The value (a string or a number) at the given key or 0 if the
 * value isn't a string or a number.
 *
 * \bug In fact, it is not a bug but... Well, here is it ;-) Lua strings
 * may have embeddend \0 insided them, whereas the real string continues.
 * This function doesn't handle the problem. There should be a subclass
 * LuaStringTable that handles a table of strings, or a LuaString class
 * that return a correct string given the index of the string in the
 * stack. Feel free to write these functions if they don't exist and you
 * need them... ;-)
 ******************************************************************************/
const char* LuaTable::getString(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);
  
  const char* result = lua_tostring(m_L, -1);
  lua_pop(m_L, 1);

  return result;
}

/******************************************************************************
 * Read a user data in the table.
 *
 * \param key The key of the element to get in the table.
 * \return A pointer to some userdata, or 0 if the value in the table is
 * not of luatype \e userdata.
 ******************************************************************************/
void* LuaTable::getUserData(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);
  
  void* result = lua_touserdata(m_L, -1);
  lua_pop(m_L, 1);

  return result;
}

/******************************************************************************
 * Read a table in the table.
 *
 * \param key The key of the element to get in the table.
 * \return A pointer on a LuaTable or 0 if there's no table at the given
 * key.
 ******************************************************************************/
LuaTable* LuaTable::getTable(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);

  LuaTable* newTable = 0;
  
  if( lua_istable(m_L, -1) )
  {
    newTable = new LuaTable(m_L, -1);
  }

  return newTable;
}

/******************************************************************************
 * The following functions add element to the table at a given key.
 * TODO: add more types...
 ******************************************************************************/

void LuaTable::addElement(const char* key, double value)
{
  lua_pushstring( m_L, key );
  lua_pushnumber( m_L, value );
  lua_settable( m_L, m_Index );
}

/******************************************************************************
 * Use this function to test if the value at the given key is nil.
 *
 * \param key The value's key in the table.
 * \return True if the value at the given key is nil.
 ******************************************************************************/
bool LuaTable::isNil(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);

  return lua_isnil(m_L, -1);
}

/******************************************************************************
 * Use this function if you think that the get*() functions of this class
 * are not efficient for your purpose. It pushes the value at the given
 * table's key on the stack. You may then use the classical lua
 * functions...
 *
 * \param key The key of the element to get in the table.
 ******************************************************************************/
void LuaTable::pushElementAt(double key)
{
  lua_pushnumber(m_L, key);
  lua_gettable(m_L, m_Index);
}

/******************************************************************************
 * Constructs a LuaTable from scratch. This creates a new table on the
 * stack.
 *
 * \param L The lua state where we are going to create the table.
 ******************************************************************************/
LuaTable* LuaTable::createTable(lua_State* L)
{
  lua_newtable(L);
  return new LuaTable(L, -1);
}

}
