/* $Id: ArkEntity.cpp,v 1.57 2003/03/16 23:15:56 mat Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkWorld.h>
#include <Ark/ArkEntity.h>
#include <Ark/ArkPath.h>
#include <Ark/ArkNetwork.h>
#include <math.h>

namespace Ark
{

   Entity::Entity (World *world) :
      m_AttachmentEnt (0)
   {
      m_Path = Path( this );
      m_ID = 0;
      m_Changed = EVERYTHING;
      m_Flags = VISIBLE|COLLISION|GRAVITY|PATHFINDING;
      
      m_World = world;
      m_WorldData = 0;
   }

   Entity::~Entity()
   {
      if (m_Flags & INWORLD)
	 m_World->Remove (this);

      for (size_t i = 0; i < m_Particles.size(); i++)
	 delete m_Particles[i];
   }
   

   /**
    * Read all informations avalaible from the given file (or the given
    * socket...). It can be used to restore the data previously written in a
    * file, or to receive data from a socket.
    */
   void
   Entity::Read (Stream &file)
   {
      unsigned char changed;
      NetReadByte(file, changed);
  
      if (changed & NAME)
	 NetReadString(file, m_Name);
      
      //cerr << "Updating entity " << m_Name << "\n";
      if (changed & POSITION)
      {
	 NetReadScalar (file, m_MState.m_Position.X);
	 NetReadScalar (file, m_MState.m_Position.Y);
	 NetReadScalar (file, m_MState.m_Position.Z);
	 //cerr << "newpos : " << m_MState.m_Position << "\n";
      }

      if (changed & PSYS)
      {
	 //cerr << "newpsys\n";
	 short bm_changed,bm_stopped;

	 NetReadShort(file, bm_changed);
	 NetReadShort(file, bm_stopped);

	 for (unsigned i = 0; i < 16; i++)
	 {
	    if (bm_changed & (1 << i))
	    {
	       String d1, d2, d3;
	       NetReadString(file, d1);
	       NetReadString(file, d2);
	       NetReadString(file, d3);

	       PsysSet (i, d1, d2, d3);
	    }
	    else if (bm_stopped & (1<<i))
	       PsysStop (i);
	 }
      }

#if 0      
      // FIXME: RECEIVE THE PATH GODDAMIT ! :)
      if (changed & PATH)
      {
	 int nsteps;
	 
	 NetReadInt(file, nsteps);
	 //cerr << "newpath " << nsteps << "\n";
	 
	 if (nsteps)
	 {
	    m_Path.Reset(); // FIXME : ??
		       
	    for (int i = 0; i < nsteps; i++)
	    {
	       Vector3 v;
	       scalar speed;
	       
	       NetReadScalar(file, v.X);
	       NetReadScalar(file, v.Y);
	       NetReadScalar(file, v.Z);
	       NetReadScalar(file, speed);
	  
	       m_Path.AddPoint (v, speed);
	    }

	    m_Path.Compute();
	    m_Path.m_MovingTarget = false;
	 }
      }
#endif

      if (changed & MODEL)
      {
	 String str;
	 NetReadString (file, str);
	 //cerr << "newmodel " << str << "\n";
	 SetModel (str);
      }
      
      if (changed & MODELSTATE)
      {
	 //cerr << "newmodelstate \n";
	 m_MState.Read (file);
      }
   }
   
   
   // Write some informations concerning the entity in the given file.
   // (an information is sent only when the bit corresponding to it is set
   // in the changed param).
   void
   Entity::Write (WriteStream &file, bool all)
   {    
      int8 changed = all ? 0xFF : m_Changed;
      NetWriteByte (file, changed);
      
      if (changed & NAME)
	 NetWriteString(file, m_Name);
      
      if (changed & POSITION)
      {
	 NetWriteScalar(file, m_MState.m_Position.X);
	 NetWriteScalar(file, m_MState.m_Position.Y);
	 NetWriteScalar(file, m_MState.m_Position.Z);
      }
      
      // First we send a bitmask of changed particle systems, and
      // stopped particle systems.
      // Then we send the strings used to create the new particle
      // systems.
      if (changed & PSYS)
      {
	 unsigned short bm_changed = 0;
	 unsigned short bm_stopped = 0;
	 size_t i;
	 
	 for (i = 0; i < m_Particles.size(); i++)
	 {
	    if (m_Particles[i] && (all || m_Particles[i]->m_Changed))
	    {
	       if (m_Particles[i]->Stopped())
		  bm_stopped |= 1 << i;
	       else
		  bm_changed |= 1 << i;
	    }
	 }
	 
	 NetWriteShort(file, bm_changed);
	 NetWriteShort(file, bm_stopped);

	 for (i = 0; i < 16; i++)
	 {
	    if (bm_stopped & (1<<i))
	       continue;

	    if (bm_changed & (1<<i) && m_Particles[i])
	    {
	       NetWriteString(file, m_Particles[i]->m_PosVel);
	       NetWriteString(file, m_Particles[i]->m_Defs);
	       NetWriteString(file, m_Particles[i]->m_Parts);
	    }
	 }
      }

#if 0      
      // FIXME: SEND THE PATH GODDAMIT ! :)
      if (changed & PATH)
      {
	 NetWriteInt(file, (int)m_Path.size());

	 for (size_t i = 0; i < m_Path.size(); i++)
	 {
	    SplinePoint &p = m_Path[i];
	    Vector3 &v = p.second;
	    
	    NetWriteScalar (file, v.X);
	    NetWriteScalar (file, v.Y);
	    NetWriteScalar (file, v.Z);

	    if (i >= 1)
	    {
	       NetWriteScalar
		  (file, (v - m_Path[i-1].second).GetMagnitude() /
		         (p.first - m_Path[i-1].first));
	    }
	    else
	       NetWriteScalar (file, 0.0);
	 }
      }
#endif

      if (changed & MODEL)
      {
	 // FIXME: Move this to ModelState::Write, and also send skin infos.
	 NetWriteString(file, m_MState.GetModel()->Name());
      }

      if (changed & MODELSTATE)
	 m_MState.Write (file);
   }
   
   // Changes the model (sets the 'model changed' flag, and re-init the
   // modelstate)
   void
   Entity::SetModel (const String &model)
   {
      m_MState.SetModel (m_World->GetCache(), model);
      m_Changed |= MODEL;
   }
   
   // Set the entity position and set the related flag
   void
   Entity::SetPosition (const Vector3 &pos)
   {
      m_MState.m_Position = pos;
      m_Changed |= POSITION;
   }

   /**
    * Set then nth particle system for this entity. Have a look at
    * the documentation of ParticleSys::ParticleSys() to learn what
    * the different parameters refer to.
    */
   void
   Entity::PsysSet (int n,
		    const Ark::String &posvel, 
		    const Ark::String &defs,
		    const Ark::String &part)
   {
      if (int(m_Particles.size()) <= n)
	 m_Particles.resize (n+1);
      
      if (m_Particles[n])
	 delete m_Particles[n];

      m_Particles[n] = new EntityPSys
	 (m_World, posvel, defs, part);

      m_Changed |= PSYS;
   }

   /**
    * Stop the nth particle set : this one will be deleted when
    * there will be no more particles in it.
    */
   void
   Entity::PsysStop (int n)
   {
      if (int(m_Particles.size()) <= n)
	 return;

      if (m_Particles[n])
	 m_Particles[n]->Stop();

      m_Changed |= PSYS;
   }
   
   /**
    * Returns true if there is a particle system numbered 'n', 
    * false otherwise.
    */
   bool
   Entity::PsysExists (int n)
   {
      if (int(m_Particles.size()) > n && m_Particles[n])
	 return true;

      return false;
   }
   
   
   /*
    * Does everything needed to update this entity, after an elapsed time of
    * "dtime". Most of times it just moves the entity according to the
    * path/trajectory sent by the server, and updates the particle systems.
    */
   void
   Entity::Update (scalar dt)
   {
      // Update position...

      if (IsAttached())
      {
	 Model *model = m_AttachmentEnt->m_MState.GetModel();
	 Attachment &attach = model->m_Attachments[m_Attachment];

	 Matrix44 &bonemtx = m_AttachmentEnt->m_MState.m_BoneMatrices
	    [attach.m_Bone];
	 
	 // Try to find out the transformation associated with the bone to 
	 // which the entity is linked in world space.
	 m_MState.m_Position = bonemtx.Transform (attach.m_Origin);

	 m_MState.m_Matrix = model->m_Skeleton->m_RootMatrix;
	 m_MState.m_Matrix.Invert();	 
	 m_MState.m_Matrix.Multiply (bonemtx);

	 // Update the modelstate
	 m_MState.Update(dt);
      }
      else
      {
	 if (m_Flags & PATHFINDING)
	 {
	    Vector3 tgt, oldpos = m_MState.m_Position; float tgtmag;
	    m_Path.Update (&m_MState.m_Position, dt);
	    
	    tgt = m_MState.m_Position - oldpos; tgt.Y = 0.0f; 
	    
	    if ((tgtmag = tgt.GetMagnitude()) > 1e-3f)
	    {
	       tgt.Scale (1.f/tgtmag);
	       scalar yaw = -(atan2f(tgt.Z, tgt.X) * 180.f / PI);
	       
	       Vector3 oldtgt = m_MState.m_Rotation.Transform(Vector3(1.f,0.f,0.f));
	       scalar y = -(atan2f(oldtgt.Z, oldtgt.X) * 180.f / PI);
	       
	       scalar diffy = yaw - y;
	       
	       while (diffy > 180.0) diffy -= 180.0;
	       while (diffy < -180.0) diffy += 180.0;
	       
	       if (fabsf(diffy) > 5.0f)
	       {
		  diffy = diffy / fabsf(diffy);
		  m_MState.m_Rotation = Quat (0.0f, y+(dt*180.0f)*diffy, 0.0f);
	       }
	    }
	 }
	    
	 // Physics (hum hum)...	 
	 if (m_Flags & GRAVITY)
	 {
	    const Vector3& pos = m_MState.m_Position;
	    Vector3 newPos = m_World->GetRestPosition(pos);
	    m_MState.m_Position = newPos;
	 }

	 // Update the modelstate
	 m_MState.Update(dt);
	 // BBox bb = m_MState.ExtractBbox ();
	 //m_MState.m_Position.Y -= (bb.m_Min.Y);
	 
	 // Everything's done : compute a matrix...
	 m_MState.ComputeMatrix();
      }

      // Update Particle systems...
      std::vector <EntityPSys*>::iterator i;
      for (i = m_Particles.begin(); i != m_Particles.end(); i++)
      {
	 if (*i == NULL)
	    continue;

	 (*i)->m_Pos = m_MState.m_Matrix.Transform((*i)->m_RelPos);
	 (*i)->Update (dt);

	 if ((*i)->Ended())
	 {
	    delete (*i);
	    *i = NULL;
	 }
      }


      // Update bounding box
      m_BBox = m_MState.ExtractBbox();
      m_BBox.Transform (m_MState.m_Matrix);
      
      return;
   }

   /** 
    * Add a message wich will be sent the next time the entity is
    * updated by a WorldUpdater.
    *  \param dest is the entity this message will be sent to.
    *  \param message is the content of the message.
    *  \param answer_list is a list of possible answers (this can
    *         be a way for a NPC to ask a question with a limited set
    *         of answers);
    */
   bool
   Entity::AddMessage (Entity *dest, const String &message,
	   std::vector<String> *answer_list)
   {
      m_OutMessages.push_back
	 (EntityMessage (dest, message, answer_list));
      return true;
   }

   /**
    * Add a collision to the entity's collision list. This function
    * is normally called by the world's DetectCollision(). 
    *   \param with is a pointer to the hit entity.
    *   \param pair is a description of the collision (bodyparts,
    *          triangle).
    *   \param potential tells wether this collision is only a bounding
    *          box one. such collisions might be ignored by the physic
    *          or simulation code, but are hints for the AI and the
    *          pathfinding.
    */
   bool
   Entity::AddCollision (Entity *with, const ColPair &pair,
			 bool potential)
   {
      assert (with != NULL);

      EntityCollision ec;
      ec.m_Collider	= with;
      ec.m_Pair		= pair;
      ec.m_Potential    = potential;

      m_Collisions.push_back (ec);
      return true;
   }
   
   /**
    * Retrieve a collision and remove if of the collision list. This
    * function is used to send "onHit" events to the scripting language.
    *   \param with will be filled by a pointer to the hit entity.
    *   \param pair will be filled by a description of the collision.
    *   \return true if there was still a collision in the collision
    *           list, false otherwise. In this second case, the two
    *           parameters remains unchanged.
    */
   bool
   Entity::PeekCollision (Entity **with, ColPair *pair,
			  bool *potential)
   {
      assert (with != NULL);
      assert (pair != NULL);

      if (m_Collisions.empty())
		  return false;

      EntityCollisionListIterator i = m_Collisions.begin();
      *with = i->m_Collider;
      *pair = i->m_Pair;
      *potential = i->m_Potential;

      m_Collisions.erase(i);
      return true;
   }

   /*
    * Attach this entity to another. If the given entity is nil, then
    * the entity is simply detached.
    *  \param ent pointer to the entity this one will be bound to.
    *  \param attachment name of the attachment point.
    */
   void
   Entity::AttachToEntity (Entity *ent, const String &attachment)
   {
      //FIXME: set a flag somewhere about a new attachment.
      
      m_AttachmentEnt = ent;
      m_Attachment = attachment;
   }

   /*
    * This function returns true if the entity is attached to another.
    * It is used by the world update function, to ensure the
    * transformation of the entity to which this one is attached is
    * computed before.
    */
   bool
   Entity::IsAttached ()
   {
      return (m_AttachmentEnt != NULL);
   }
   
   void
   Entity::SetGoal (const Vector3 &goal)
   {
      m_Goal = EntityGoal (goal);
      SetChanged (GOAL);
      return;
   }
   
   void Entity::SetGoal (Entity *goal)
   {
      m_Goal = EntityGoal(goal);
      SetChanged (GOAL);
      return;
   }
   
   // This function should be called when an entity is not used any more,
   // instead of destroying it. If the entity belongs to a World, it is marked
   // as dead, and will be destroyed at the next world update. If not, it is
   // immediatly destroyed.
   void
   Entity::SetDead ()
   {
      if ((m_Flags & INWORLD) == 0)
	 delete this;
      else
	 m_Flags |= DEAD;
   }
   
}
