/**********************************************************************
 ** DataLog class: Used for logging user-input data such as suggestions
 **                or bugs found
 **    
 ** Last reviewed: 
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef DATA_LOG_C
#define DATA_LOG_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "logs.h"
#include "utils.h"
#include "data_log.h"
#include "editor.h"
#include "inp_handler.h"
#include "player.h"
#include "newfuncts.h"

/***********************************************************************
 ** Data_Log (constructor) - Opens the logfile by a specified name
 **
 ** Parameters: log_name - the name of the log we are opening
 **             file_name - the log filename to use
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Data_Log::Data_Log(char *log_name, char *file_name)
{
   valid = 0;
   if (open_log(file_name) <= 0)
   {
      printf("Error - could not open the data log %s\n", log_name);
      return;
   }
   data_log_name = log_name;
   fprintf(the_log, "&+W<%s>&+b ** Mud Starting up **&*\n", get_time_str());
   fflush(the_log);
   valid = 1;
}


/***********************************************************************
 ** ~Data_Log (destructor)  - empty (closing actions located in Logs)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Data_Log::~Data_Log()
{
      /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return;

   fprintf(the_log, "&+W<%s>&+b ** Mud Closing down **&*\n", get_time_str());
   fflush(the_log);

}


/***********************************************************************
 ** log_err - logs an entry in the data log
 **
 ** Parameters: player_name - the player name who submitted this
 **             loc_name - the location name that the player is at
 **             data_str - the data string to print
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int Data_Log::log_data(char *player_name, char *player_loc, char *data_str)
{

   /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return -1;

   fprintf(the_log, "&+W<%s>&* &+m%s&* (&+g%s&*)\n%s", get_time_str(), 
                                   player_name, player_loc, data_str);
   fflush(the_log);
   return 1;
}


/***********************************************************************
 ** user_add_data - starts the user adding data to the datafile
 **
 ** Parameters: the_player - the player who is adding the data
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int Data_Log::user_add_data(Player *the_player)
{
   Editor *input_editor;
   Inp_Handler *player_handler;   

   Strings *the_data;
   char   *holder;
   int    results;

   if (the_player == NULL)
      return -1;

   the_data = new Strings();
   input_editor = new Editor();
   results = input_editor->start_editor((MudObject *) the_player, the_data);
   if (!STRCASECMP(data_log_name.str_show(), "Bug Log"))
      holder = "bug";
   else if (!STRCASECMP(data_log_name.str_show(), "Suggestions Log"))
      holder = "suggest";
   else
      return -1;

   player_handler = the_player->get_input_handler();
   player_handler->add_pop_function(data_log_on_pop, the_player->get_name(), 
                            the_player->get_location(), holder, the_data);
   return results;
}



/***********************************************************************
 ** archive_datalog - archives the data log and opens a new one
 **
 ** Parameters: file_name - the name of the file to open
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int Data_Log::archive_datalog(char *file_name)
{
   int results; 

   fprintf(the_log, "<%s> ** Archiving DataLog **\n", get_time_str());
   fflush(the_log);
   
   if ((results = archive_log(file_name)) != -1)
   {
      if (open_log(file_name) <= 0)
      {
         printf("Error - could not open the datalog\n");
	 valid = 0;
         return -1;
      }
      fprintf(the_log, "<%s> ** Data log reopened after archive **\n", 
                                                          get_time_str());
      fflush(the_log);
      valid = 1;
   }
   return results;
}


#endif
