/**********************************************************************
 * cidr_net.h                                                March 2001
 * Horms                                             horms@verge.net.au
 *
 * aggregate
 * CIDR network aggregation and filtering
 * Copyright (C) 1999-2002  Horms
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 *
 **********************************************************************/


#ifndef CIDR_NET_FRUB
#define CIDR_NET_FRUB

#define IP_STRING_LENGTH 16
#define IP_BITS 32

#include "int.h"

#include "ip_address.h"
#include "prefix.h"

typedef uint32 cidr_flag_t;

typedef struct {
  ip_address_t address;
  ip_address_t top_address;
  prefix_t prefix;
} cidr_net_t;

#define DEFAULT_NORECENT 7

/* Flags for display_cidr_net*/
#define CIDR_ADDRESS_RANGE       0x1      /* Display as an address range
                                           * e.g.: 10.0.0.1 - 10.0.0.255 */
#define CIDR_NET_NETMASK         0x2      /* Display as an address and netmask
                                           * e.g.: 10.0.0.0/255.255.255.0 */
#define CIDR_NET_PREFIX          0x4      /* Display as an address and prefix
                                           * e.g.: 10.0.0.0/24 */

#define CIDR_NET_STR_LEN 2 * (IP_STRING_LENGTH-1) + 4


/**********************************************************************
 * cidr_net_create
 * Initialise a cidr_net and seed values
 * pre: a: base address of network
 *      p: prefix of network
 * post: n is initialise and values are seeded
 *       NULL on error
 **********************************************************************/

cidr_net_t *cidr_net_create(ip_address_t a, prefix_t p);


/**********************************************************************
 * cidr_net_assign
 * Assign values to a cidr_net structure
 * pre: n: pointer to cidr_net structure
 *      a: base address of network
 *      p: prefix of network
 * post: n is initialised
 *       NULL if n is NULL
 **********************************************************************/

cidr_net_t *cidr_net_assign(cidr_net_t *n, ip_address_t a, prefix_t p);


/**********************************************************************
 * cidr_net_destroy
 * Destory a cidr_net
 * pre: n: pointer to cidr_net to destroy
 * post: n is destroyed
 **********************************************************************/

void cidr_net_destroy(cidr_net_t *n);


/**********************************************************************
 * cidr_net_to_a
 * Print to stdout the network in address/prefix notation, where
 * address is in dotted quad notation
 * pre: n: network to display
 *      str: prealocated string to "render" dotted quad noation into
 *      minimim_prefix: Don't display networks with a prefix less than
 *                      this value. Valid values 0-32. 
 *                      -1 for no mimimium prefix.
 *      maximim_prefix: Don't display networks with a prefix greater than
 *                      this value. Valid values 0-32. 
 *                      -1 for no maximium prefix.
 *      flag: as per cidr_net.h
 * post: network is displayed to stdout
 *       str contains the network in dotted quad notation
 * Note: Result is rendered into a static buffer
 **********************************************************************/

char *cidr_net_to_a(
  cidr_net_t *n, 
  prefix_t minimum_prefix,
  prefix_t maximum_prefix,
  cidr_flag_t flag
);


/**********************************************************************
 * cidr_net_distance
 * Find the distance between two networks base address
 * pre: a, b: networks to find the distance between
 * return: absolute diffrence of the base address of a and b
 **********************************************************************/

ip_address_t cidr_net_distance(cidr_net_t *a, cidr_net_t *b);


/**********************************************************************
 * cidr_net_cmp
 * Compare two cidr networks 
 * pre: a, b: networks to compare
 * return: 1 if a and b are the same
 *         0 if and are be are not the same
 *         -1 on error
 **********************************************************************/

int cidr_net_cmp(cidr_net_t *a, cidr_net_t *b);


/**********************************************************************
 * cidr_generate_pow_2_lookup
 * Generate an array with IP_BITS elements where element n = 2^n
 * pre: none
 * post: @bit_loookup is seeded (or seedy as the case may be)
 **********************************************************************/

void cidr_generate_pow_2_lookup(void);


/**********************************************************************
 * cidr_log_2
 * Arghh this is really nasty, I am sure there is a better way to do 
 * this. 
 * Get log base 2 of a IP_BITS bit integer, rounded down to the nearest
 * integer
 * Works on IP_BITSbit numbers
 * pre: address: address to find log base 2 of
 * post: integer log base 2
 **********************************************************************/

int cidr_log_2 (ip_address_t address);


/**********************************************************************
 * cidr_netmask
 * Calculate the netmask given a prefix
 * pre: prefix: prefix to find the netmask off
 * return: netmask in decimal
 **********************************************************************/

ip_address_t cidr_netmask(prefix_t prefix);
  

/**********************************************************************
 * cidr_prefix
 * Arghh this is really nasty, I am sure there is a better way to do 
 * this.
 * Get the network prefix (IP_BITS-number of trailing zeros)
 * Works on IP_BITSbit numbers
 * pre: address: the address to find the prefix of
 * post: number of trailing zeros.
 **********************************************************************/

prefix_t cidr_prefix (ip_address_t address);


/**********************************************************************
 * cidr_classful_prefix
 * Given an ip address find the classful network prefix length
 * pre: address: the address to find the prefix of
 * post: The number of trailing zeros, rounded to the nearest 
 *       classful prefix length. That is one of 0, 8, 16, 24 or 32.
 **********************************************************************/

prefix_t cidr_classful_prefix(ip_address_t address);

#endif
