/**********************************************************************
 * ip_address.c                                              March 2001
 * Horms                                             horms@verge.net.au
 *
 * aggregate
 * CIDR network aggregation and filtering
 * Copyright (C) 1999-2002  Horms
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 *
 **********************************************************************/


#include "ip_address.h"
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>


/**********************************************************************
 * ip_address_dotted_quad_to_decimal
 * Convert an IP address given as a string into decimal
 * pre: alleged_ip: string representing ip address
 * post: 0 on success, result contains value
 *       -1 on error
 **********************************************************************/

int ip_address_dotted_quad_to_decimal(
  char *alleged_ip, 
  ip_address_t *decimal_ip
){
  size_t length;
  ip_address_t octet_multipler;
  octet_t   octet_digit_multipler;
  int       current_digits;
  int       nooctet;
  octet_t   decimal_octet;
  char      *current;

  /*Get length and check it isn't too long*/
  if((length=strlen(alleged_ip))>IP_STRING_LENGTH-1){
    return(-1);
  }

  octet_multipler        =(ip_address_t)1;
  octet_digit_multipler  =  (octet_t)1;
  current_digits         =           0;
  nooctet                =           0;
  *decimal_ip            =(ip_address_t)0;
  decimal_octet          =  (octet_t)0;

  for(current=alleged_ip+length-1;current>alleged_ip-2;current--){
    if(isdigit(*current)){
      current_digits++;
      decimal_octet+=((octet_t)(*current-'0')*octet_digit_multipler);
      octet_digit_multipler*=(octet_t)10;
    }
    else if(*current=='.' || current<alleged_ip){
      if(
        current_digits==0          ||
        current_digits>3           ||
        decimal_octet>(octet_t)255 ||
        ++nooctet>4
      ){ 
        return (-1); 
      }
      *decimal_ip+=(ip_address_t)(decimal_octet*octet_multipler);
      octet_multipler<<=8;
      octet_digit_multipler=(octet_t)1;
      decimal_octet=(octet_t)0;
      current_digits=0;
    }
    else {
      return(-1);
    }
  }

  
  for(;nooctet<4;nooctet++){
    *decimal_ip<<=8;
  }

  return(0);
}


/**********************************************************************
 * ip_address_decimal_to_dotted_quad
 * Turn an IP given as a decimal to a dotted quad
 * pre: ip_address: string representing IP address
 * post: NULL if an error occurs
 *       dotted quad representation of IP address otherwise
 * Note: result is rendered into a static buffer
 **********************************************************************/

static char __ip_address_str[IP_STRING_LENGTH];

char *ip_address_decimal_to_dotted_quad(ip_address_t ip_address){
  snprintf(
    __ip_address_str,
    IP_STRING_LENGTH,
    "%d.%d.%d.%d",
    (ip_address>>24)&255,
    (ip_address>>16)&255,
    (ip_address>>8)&255,
    ip_address&255
  );

  return(__ip_address_str);
}
