#include "CalendarEventView.h"
#include <math.h>

#define CEV_MINCOLSIZE 60
#define CEV_SCROLLBORDER 30
#define CEV_BASELINESIZE 3
#define CEV_TXTBORDER 3
#define CEV_STEPSECONDS (_minutesInStep * 60)
#define CEV_UNITSECONDS (CEV_STEPSECONDS * _stepsInUnit)
#define CEV_VIEWSECONDS (CEV_UNITSECONDS * _unitsInView)

#define CEV_UNITHEIGHT (_stepsInUnit * _stepHeight)
#define CEV_VIEWHEIGHT (CEV_UNITHEIGHT * _unitsInView)
#define CEV_VIEWWIDTH (NSWidth(_bounds) - _cevOffset)

#include "Tint.h"

@interface CalendarEventView (Private)
- (void) _sortViews;
- (void) _createEvent;
- (void) _adjustColumn;
- (void) _updateSelect;
- (void) _frameChanged;
- (void) _dateChanged;
@end

@implementation CalendarEventView (Private)
static NSComparisonResult cev_compare(id a, id b, id c)
{
	if ([a isSelected])
	{
		return NSOrderedDescending;
	}
	else if ([b isSelected])
	{
		return NSOrderedAscending;
	}
	
	return NSOrderedSame;
}

- (void) _sortViews
{
	[self sortSubviewsUsingFunction:&cev_compare
							context:nil];

}

- (void) _createEvent
{
	NSTimeInterval start,stop;
	start = _startTimeInterval;
	stop = _endTimeInterval;
	id model;

	_startTimeInterval = _endTimeInterval = 0;

	if (start != stop)
	{

		model = [_delegate createEventWithStartDate:[[_date addTimeInterval:start] earlierDate:[_date addTimeInterval:stop]]
									   timeInterval:fabs(start - stop)];
		[self selectModel:model];
		[self editSelectedCell];
	}

}

- (void) _updateSelect
{
	NSEnumerator *en;
	id cell;
	en = [[self subviews] objectEnumerator];
	while (cell = [en nextObject])
	{
		if ([cell model] == [_delegate selectedModel])
		{
			[cell setSelected:YES];
		}
		else
		{
			[cell setSelected:NO];
		}
	}
}

- (void) editSelectedCell
{
	NSEnumerator *en;
	id cell;
	en = [[self subviews] objectEnumerator];
	while (cell = [en nextObject])
	{
		if ([cell model] == [_delegate selectedModel])
		{
			[cell edit];
			break;
		}
	}
}

- (void) _adjustColumn
{
	NSMutableArray *newA;
	NSEnumerator *en;
	id cell;
	NSCalendarDate *lateDate;
	NSArray *views;
	CREATE_AUTORELEASE_POOL(p);

	/* split objects into groups */
	views = [[self subviews] sortedArrayUsingSelector:@selector(compare:)];
	en = [views objectEnumerator];
	cell = [en nextObject];
	while (cell != nil)
	{
		if ([[_date addTimeInterval:CEV_VIEWSECONDS] compare:[cell startDate]] <= NSOrderedSame || 
			[_date compare:[cell stopDate]] >= NSOrderedSame)
		{
			[cell hide];
		//	[cell removeFromSuperview];
			cell = [en nextObject];
			continue;
		}
		[cell calculateFrameHeight];

		newA = [NSMutableArray array];
		[newA addObject:cell];
		lateDate = [cell stopDate];

		while ((cell = [en nextObject]))
		{
			if ([[_date addTimeInterval:CEV_VIEWSECONDS] compare:[cell startDate]] <= NSOrderedSame || 
						 [_date compare:[cell stopDate]] >= NSOrderedSame)
			{
				[cell hide];
		//FIXME		[cell removeFromSuperview];
				continue;
			}
			[cell calculateFrameHeight];



			if ([lateDate compare:[cell startDate]] == NSOrderedDescending)
			{
				[newA addObject:cell];
				lateDate = [[cell stopDate] laterDate:lateDate];
			}
			else
			{
				break;
			}
		}


		/* process each group */
		{
			NSMutableArray *columns;
			NSEnumerator *en2;
			CalendarEventCell *nCell, *lCell;
			int i,count;

			columns = [NSMutableArray array];
			count = (NSWidth([self bounds]) - _cevOffset)/CEV_MINCOLSIZE;
			count = count + count;

			for (i = 0; i < count; i++)
			{
				[columns addObject:[NSMutableArray array]];
			}

			en2 = [newA objectEnumerator];

			while ((nCell = [en2 nextObject]))
			{
				for (i = 0; i < count; i++)
				{
					id col = [columns objectAtIndex:i];
					lCell = [col lastObject];
					if (lCell == nil ||
							[[lCell stopDate] compare:[nCell startDate]] <= NSOrderedSame)
					{
						[col addObject:nCell];
						break;
					}
				}
			}

			int fullcount = count;

			for (count = count/2; count > 0 && [[columns objectAtIndex:count-1] count] == 0; count--);

			float opWidth = (NSWidth([self bounds]) - _cevOffset)/count;


			while (count)
			{
				en2 = [[columns objectAtIndex:count-1] objectEnumerator];
				while ((nCell = [en2 nextObject]))
				{
					NSRect newFrame;
					newFrame = [nCell frame];
					newFrame.origin.x = _cevOffset + opWidth * (count - 1) + CEV_BASELINESIZE;
					newFrame.size.width = opWidth - CEV_BASELINESIZE * 2;

					[nCell setFrame:newFrame];
//					[self addSubview:nCell];
				}
				count--;
			}


			/* round 2 hack */
			for (count = fullcount; count > fullcount/2 && [[columns objectAtIndex:count-1] count] == 0; count--);

			while (count > fullcount/2)
			{
				en2 = [[columns objectAtIndex:count-1] objectEnumerator];
				while ((nCell = [en2 nextObject]))
				{
					NSRect newFrame;
					newFrame = [nCell frame];
					newFrame.origin.x = _cevOffset + opWidth * ((count - fullcount/2) - 1) + CEV_BASELINESIZE + opWidth/2;
					newFrame.size.width = opWidth - CEV_BASELINESIZE * 2;

					[nCell setFrame:newFrame];
//					[self addSubview:nCell];
				}
				count--;
			}

		}
	}
	[self _sortViews];
	RELEASE(p);
}

- (void) _dateChanged
{
	int i;
	NSCalendarDate *runDate;
	NSTimeInterval sinceTime;
	int lastDay;

	/* may want to recalcuate font size in future */

	lastDay = 0;

	if (_showIndex)
	{
		_cevOffset = 0;
		ASSIGN(_timeStr, [NSMutableArray array]);
		for(i = 0, sinceTime = 0; i <= _unitsInView; i++, sinceTime += CEV_UNITSECONDS)
		{
			runDate = [_date addTimeInterval:sinceTime];

			NSSize strsize;
			NSMutableAttributedString *str;

			str = [[NSMutableAttributedString alloc]
						initWithString:[NSString stringWithFormat:@"%02d:%02d",
											[runDate hourOfDay],
											[runDate minuteOfHour]]];

			AUTORELEASE(str);

			strsize = [str size];
			if (_cevOffset < strsize.width)
			{
				_cevOffset = strsize.width;
			}

			if ([runDate dayOfMonth] != lastDay)
			{
				id hackDay[] = {@"Sun",@"Mon",@"Tue",@"Wed",@"Thu",@"Fri",@"Sat"};

				id dateStr;
				dateStr = [[NSMutableAttributedString alloc] initWithString:[NSString stringWithFormat:@"%@\n",hackDay[[runDate dayOfWeek]]]];
				AUTORELEASE(dateStr);
				[dateStr addAttribute:NSFontAttributeName
								value:[NSFont systemFontOfSize:8]
								range:NSMakeRange(0,[dateStr length])];



				[str insertAttributedString:dateStr
									atIndex:0];

				dateStr = [[NSMutableAttributedString alloc] initWithString:[NSString stringWithFormat:@"\n%d/%d", [runDate monthOfYear], [runDate dayOfMonth]]];
				AUTORELEASE(dateStr);
				[dateStr addAttribute:NSFontAttributeName
								value:[NSFont systemFontOfSize:8]
								range:NSMakeRange(0,[dateStr length])];

				[str appendAttributedString:dateStr];

											
				lastDay = [runDate dayOfMonth];
			}

			[str addAttribute:NSForegroundColorAttributeName
						value:[NSColor blackColor]
						range:NSMakeRange(0,[str length])];
			[_timeStr addObject:str];


			str = [str mutableCopy];
			AUTORELEASE(str);

			[str addAttribute:NSForegroundColorAttributeName
						value:[NSColor whiteColor]
						range:NSMakeRange(0,[str length])];
			[_timeStr addObject:str];
		}
		_cevOffset += CEV_TXTBORDER * 2;
	}

	[self update];
}

- (void) viewWeek
{
	_minutesInStep = 360;
	_stepsInUnit = 4; 
	_unitsInView = 7;
	NSRect r = _frame;
	r.size.height = 1000;
	[self setFrame:r];
	[self setNeedsDisplay:YES];
}

- (void) viewYear
{
}

- (void) viewDay
{
	_minutesInStep = 15;
	_stepsInUnit = 4;
	_unitsInView = 24;
	NSRect r = _frame;
	r.size.height = 800;
	[self setFrame:r];
	[self setNeedsDisplay:YES];
}

- (void) viewMonth
{
	_minutesInStep = 1440;
	_stepsInUnit = 7;
	_unitsInView = 4;
	NSRect r = _frame;
	r.size.height = 500;
	[self setFrame:r];
	[self setNeedsDisplay:YES];
}

- (void) _frameChanged
{
	NSRect b = [self bounds];

	//test
	_showIndex = YES;

	_stepHeight = (NSHeight(b) - CEV_BORDERSIZE * 2) /
		(_unitsInView * _stepsInUnit);

	[self _dateChanged];
}


@end


@implementation CalendarEventView

- (id) initWithFrame:(NSRect)frame
{
	[super initWithFrame:frame];

	_workTimeStart = 3600 * 6;
	_workTimeStop = 3600 * 21;
	_cevOffset = 0;

	[self setDate:[NSCalendarDate calendarDate]];
	[self setFrame:frame];
	[self viewWeek];


	return self;
}

- (void) update
{
	NSArray *array = [_delegate allEventsDuringDate:_date
											andDate:[_date addTimeInterval:CEV_VIEWSECONDS]];

	id en,view;

	/* clear all view TODO optimize this */
	en = [[self subviews] objectEnumerator];
	while ((view = [en nextObject]))
	{
		[view removeFromSuperview];
	}

	CalendarEventModel *model;

	en = [array objectEnumerator];
	while ((model = [en nextObject]))
	{
		id cell = [[CalendarEventCell alloc] initWithOwner:self
													 model:model];
		[self addSubview:cell];
		AUTORELEASE(cell);
	}
	[self _adjustColumn];

}

- (void) selectModel:(id)model
{
	[_delegate selectModel:model];
	[self _updateSelect];
}


/** overidden view methods **/
-(void) setFrame:(NSRect)f
{
	[super setFrame:f];
	[self _frameChanged];
}

- (void) drawRect:(NSRect)r
{
	int i,j;
	float hline;

	r = [self bounds];

	/* fill all space. TODO optimize*/
	[[NSColor grayColor] set];
	NSRectFill(r);

	[[NSColor textBackgroundColor] set];

	hline = CEV_BORDERSIZE;

	NSRectFill(
			NSMakeRect(	_cevOffset,
						hline,
						CEV_VIEWWIDTH,
						NSHeight(r) - 2 * CEV_BORDERSIZE));

	[[NSColor lightGrayColor] set];
	/* fill bottom */
	/*
	NSRectFill(
			NSMakeRect(	_cevOffset,
						hline,
						NSWidth(r) - _cevOffset,
						_stepHeight *
						(CEV_VIEWSECONDS - _workTimeStop) /
						(60 * _minutesInStep)));
						*/

	/* fill top */
	/*
	NSRectFill(
			NSMakeRect(	_cevOffset,
						hline +
							_stepHeight *
							(CEV_VIEWSECONDS - _workTimeStart) /
							(60 * _minutesInStep),
						NSWidth(r) - _cevOffset,
						_stepHeight * (_workTimeStart)/CEV_STEPSECONDS));
						*/

	hline = CEV_VIEWHEIGHT + CEV_BORDERSIZE;

	if (_showIndex)
	{
		for(i = 0; i <= _unitsInView; i++, hline -= CEV_UNITHEIGHT)
		{
			NSAttributedString *str;
			str = [_timeStr objectAtIndex:i*2];
			NSSize sz = [str size];
			/*shadow*/
			if (str != nil)
			{
				[str drawAtPoint:NSMakePoint(_cevOffset - sz.width - CEV_TXTBORDER + 1, hline - sz.height/2 - 1)];
			}
			str = [_timeStr objectAtIndex:(i*2)+1];
			if (str != nil)
			{
				[str drawAtPoint:NSMakePoint(_cevOffset - sz.width - CEV_TXTBORDER, hline - sz.height/2)];
			}
		}
	}

	PSnewpath();
	PSsetlinewidth(0.5);

	hline = CEV_BORDERSIZE;
	for(i = 0; i < _unitsInView; i++)
	{
		hline += _stepHeight;
		for (j = 0; j < (_stepsInUnit - 1); j++, hline += _stepHeight)
		{
			PSmoveto(_cevOffset, hline);
			PSlineto(_cevOffset + CEV_BASELINESIZE, hline);
		}
	}
	[[NSColor blackColor] set];
	PSmoveto(_cevOffset,NSMinY(r));
	PSlineto(_cevOffset,NSMaxY(r));
	PSstroke();
	PSnewpath();

	hline = CEV_BORDERSIZE;

	for(i = 0; i <= _unitsInView; i++, hline += CEV_UNITHEIGHT)
	{
		PSmoveto(_cevOffset, hline);
		PSlineto(NSMaxX(r), hline);
	}
	[[NSColor blackColor] set];
	/*
	{
		float pat[] = {1,6};
		PSgsave();
		PSsetdash(pat,2,0);
		PSstroke();
		PSgrestore();
	}
	*/
	PSstroke();

	/* draw creating tint */
	if (_startTimeInterval != _endTimeInterval)
	{

		float st,sz;

		if (_startTimeInterval < _endTimeInterval)
		{
			st = _endTimeInterval;
			sz = _startTimeInterval;
		}
		else
		{
			st = _startTimeInterval;
			sz = _endTimeInterval;
		}

		sz = st - sz;


		NSCalendarDate *_sdate;
		NSCalendarDate *_edate;
		NSMutableAttributedString *str;
		NSSize size;

		if (_startTimeInterval < _endTimeInterval)
		{
			_sdate = [_date addTimeInterval:_startTimeInterval];
			_edate = [_date addTimeInterval:_endTimeInterval];
		}
		else
		{
			_sdate = [_date addTimeInterval:_endTimeInterval];
			_edate = [_date addTimeInterval:_startTimeInterval];
		}

		str = [[NSMutableAttributedString alloc]
			initWithString:[NSString stringWithFormat:@"%d/%d %02d:%02d - %d/%d %02d:%02d",[_sdate monthOfYear],[_sdate dayOfMonth],[_sdate hourOfDay],[_sdate minuteOfHour],[_edate monthOfYear],[_edate dayOfMonth],[_edate hourOfDay],[_edate minuteOfHour]]];
		size = [str size];


		if ((_stepHeight * sz/CEV_STEPSECONDS) < size.height)
		{
			sz = CEV_STEPSECONDS * size.height /_stepHeight ;
		}


		hline = CEV_VIEWHEIGHT + CEV_BORDERSIZE;

		[tint setRect:NSMakeRect(_cevOffset + CEV_BASELINESIZE * 4 + (_stepHeight/2),
				hline - (_stepHeight * st/CEV_STEPSECONDS) + 1,
				NSWidth(r) - _cevOffset - _stepHeight - CEV_BASELINESIZE * 8,
				_stepHeight * sz/CEV_STEPSECONDS - 2)
			   radius:_stepHeight/2];

#if 0
		PSnewpath();
		PSmoveto(_cevOffset + CEV_BASELINESIZE/2 + (_stepHeight/2),
				hline - (_stepHeight * st/CEV_STEPSECONDS));
		PSrcurveto(
				-_stepHeight/2, 0,
				-_stepHeight/2 ,_stepHeight/2,
				-(_stepHeight/2), (_stepHeight/2));

		PSrlineto(0, (_stepHeight * sz/CEV_STEPSECONDS) - _stepHeight);
		PSrcurveto(
				0,_stepHeight/2,
				_stepHeight/2,_stepHeight/2,
				(_stepHeight/2), (_stepHeight/2));

		PSrlineto(NSWidth(r) - _cevOffset - _stepHeight - CEV_BASELINESIZE, 0);
		PSrcurveto(
				_stepHeight/2, 0,
				_stepHeight/2 , -_stepHeight/2,
				(_stepHeight/2), -(_stepHeight/2));

		PSrlineto(0, (-_stepHeight * sz/CEV_STEPSECONDS) + _stepHeight);
		PSrcurveto(
				0,-_stepHeight/2,
				-_stepHeight/2,-_stepHeight/2,
				
				-(_stepHeight/2), -(_stepHeight/2));

		PSclosepath();
		[[NSColor blueColor] set];
		{
			float pat[] = {3,3};
			PSgsave();
			PSsetalpha(1);
			PSsetlinewidth(1);
			PSsetdash(pat,2,0);
			PSstroke();
			PSgrestore();
		}
		PSsetalpha(0.3);
		//PSfill();

#endif

		[str addAttribute:NSForegroundColorAttributeName
					value:[NSColor blueColor]
					range:NSMakeRange(0,[str length])];

		[str addAttribute:NSFontAttributeName
					value:[NSFont boldSystemFontOfSize:10]
					range:NSMakeRange(0,[str length])];
		if (_startTimeInterval < _endTimeInterval)
		{
			[tint setStr:str
				   point:NSMakePoint(_cevOffset + CEV_VIEWWIDTH/2 - size.width/2,
						hline - (_stepHeight * st/CEV_STEPSECONDS))];
#if 0
			[str drawAtPoint:
				NSMakePoint(_cevOffset + CEV_VIEWWIDTH/2 - size.width/2,
						hline - (_stepHeight * st/CEV_STEPSECONDS))];
#endif
		}
		else
		{
			[tint setStr:str
				   point:NSMakePoint(_cevOffset + (NSWidth(r) - _cevOffset)/2 - size.width/2,
					hline + (_stepHeight * (sz-st)/CEV_STEPSECONDS) - size.height)];
#if 0
			[str drawAtPoint:NSMakePoint(_cevOffset + (NSWidth(r) - _cevOffset)/2 - size.width/2,
					hline + (_stepHeight * (sz-st)/CEV_STEPSECONDS) - size.height)];

#endif
		}
		RELEASE(str);

		
	}
}

- (void) mouseDown:(NSEvent *)event
{
	unsigned int eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask | NSLeftMouseDraggedMask | NSFlagsChangedMask;
	NSEventType eventType = [event type];
	unsigned flags = [event modifierFlags];

	if ([self hitView:event] == self)
	{
		_startTimeInterval = [self hitTime:event];
		NSLog(@"%g",_startTimeInterval);
		_endTimeInterval = _startTimeInterval;

		tint = AUTORELEASE([[Tint alloc] initWithFrame:[self bounds]]);

		[self selectModel:nil];

		[self addSubview:tint];

		while (eventType != NSLeftMouseUp)
		{

			[self setNeedsDisplay:YES];
			event = [NSApp nextEventMatchingMask:eventMask
									   untilDate:nil
										  inMode:NSEventTrackingRunLoopMode
										 dequeue:YES];
			eventType = [event type];

			NSTimeInterval t = [self hitTime:event];
			if (fabs(t - _startTimeInterval) >= (BASEMINUTES * 60))
			{
				_endTimeInterval = t;
			}
			[_super_view autoscroll:event];

		}
		[tint removeFromSuperview];

		tint = nil;

		[self _createEvent];
	}
	else
	{
		id startDate,stopDate;
		NSCalendarDate *newStartDate;
		NSCalendarDate *newStopDate;

		startDate = stopDate = nil;
		CalendarEventCell * theCell = [self hitView:event];
		NSPoint p = [theCell convertPoint:[event locationInWindow]
								 fromView:nil];
		BOOL resizing = YES;

		if (p.y > _stepHeight || p.y > NSHeight([theCell frame])/2)
		{
			resizing = NO;
		}
		
		/* select the cell */
		[self _sortViews];

		ASSIGN(startDate,[theCell startDate]);
		ASSIGN(stopDate,[theCell stopDate]);
		newStartDate = startDate;
		newStopDate = stopDate;

		NSRect origFrame = [theCell frame];

		NSTimeInterval downTime,dragTime;

		downTime = [self hitTime:event];
		while (eventType != NSLeftMouseUp)
		{

			[self setNeedsDisplay:YES];

			dragTime = [self hitTime:event];

			newStartDate = [startDate addTimeInterval:dragTime - downTime];
			newStopDate = [stopDate addTimeInterval:dragTime - downTime];

			event = [NSApp nextEventMatchingMask:eventMask
									   untilDate:nil
										  inMode:NSEventTrackingRunLoopMode
										 dequeue:YES];


			if ([newStopDate compare:_date] == NSOrderedDescending)
			{
				if ([newStartDate compare:[_date addTimeInterval:CEV_VIEWSECONDS]] == NSOrderedAscending)
				{
					if (resizing)
					{
						[theCell setTimeInterval:[newStopDate timeIntervalSinceDate:startDate]];
					}
					else
					{
						[theCell setStartDate:newStartDate
								 timeInterval:[newStopDate timeIntervalSinceDate:newStartDate]];
					}

				}
			}

			[_super_view autoscroll:event];

			eventType = [event type];

		}

		if ([newStartDate isEqualToDate:startDate] &&
			 [newStopDate isEqualToDate:stopDate])
		{
			[self selectModel:[theCell model]];
		}

		RELEASE(startDate);
		RELEASE(stopDate);

		[self _adjustColumn];

	}

	[self setNeedsDisplay:YES];
}

/*****/


- (void) _scrollToDayStart // DIRTY TODO
{
	[self scrollPoint:NSMakePoint(0,NSMaxY(_bounds))];

	//[self scrollPoint:NSMakePoint(0,NSMinY(_bounds) + NSHeight(_bounds)/(25 * 2) + _stepHeight * (DAYSECONDS - _workTimeStart)/(60*BASEMINUTES) - NSHeight([_super_view frame]) + _stepHeight * 4)];
}

- (void) viewSelectedCell
{
	NSEnumerator *en;
	id cell;
	en = [[self subviews] objectEnumerator];
	while (cell = [en nextObject])
	{
		if ([cell model] == [_delegate selectedModel])
		{
			[self viewCell:cell];
		}
	}
}

- (void) viewCell:(CalendarEventCell *)cell
{
	NSPoint newPoint = NSZeroPoint;

	newPoint.y = NSMaxY([cell frame]) + CEV_SCROLLBORDER;
	newPoint.y -= NSHeight([_super_view frame]);

	[self scrollPoint:newPoint];
}

- (void) dealloc
{
	[super dealloc];
}

- (void) setDate:(NSCalendarDate *)aDate
{
	ASSIGN(_date, [NSCalendarDate dateWithYear:[aDate yearOfCommonEra]
										 month:[aDate monthOfYear]
										   day:[aDate dayOfMonth]
										  hour:0
										minute:0
										second:0
									  timeZone:[aDate timeZone]]);
	[self _dateChanged];
	[self setNeedsDisplay:YES];

}

- (NSView *) hitView:(NSEvent *)event
{
	id view;
	NSPoint p = [_super_view convertPoint:[event locationInWindow]
								 fromView:nil];
	view = [self hitTest:p];

	while([view isKindOfClass:[CalendarEventCell class]] != YES)
	{
		view = [view superview];
		if (view == nil)
		{
			return self;
		}
	}
	return view;
}

/* return seconds of BASEMINUTES */

- (NSTimeInterval) hitTime:(NSEvent *)event
{
	NSPoint p = [self convertPoint:[event locationInWindow]
						  fromView:nil];

	p.y -= CEV_BORDERSIZE;
	return ((_stepsInUnit * _unitsInView) - floor((p.y + (_stepHeight/2))/ _stepHeight))
		* CEV_STEPSECONDS;

}


- (float) stepHeight
{
	return _stepHeight;
}

- (float) numberOfMinutesInStep
{
	return _minutesInStep;
}


- (float) numberOfStepsInUnit
{
	return _stepsInUnit;
}

- (float) numberOfUnits
{
	return _unitsInView;
}

- (void) calendarEventCellWillBeFreed:(id)sender
{
	if (sender == _selectedCell)
	{
		_selectedCell = nil;
	}
}

- (id) selectedCell
{
	return _selectedCell;
}

#if 0
- (void) mouseUp:(NSEvent *)event
{
	[self _createEvent];
	_startTimeInterval = _endTimeInterval = 0;
	[self setNeedsDisplay:YES];
}

- (void) mouseDragged:(NSEvent *)event
{
	NSTimeInterval t = [self hitTime:event];
	if (fabs(t - _startTimeInterval) >= (BASEMINUTES * 60))
	{
		_endTimeInterval = t;
	}

	[self setNeedsDisplay:YES];
}
#endif

/*
- (NSView *)hitTest:(NSPoint)aPoint
{
	return self;
}
*/

- (id) delegate
{
	return (_delegate);
}

- (void) setDelegate:(id)del
{
	ASSIGN(_delegate, del);
}

- (NSCalendarDate *) startDate
{
	return _date;
}

- (NSCalendarDate *) stopDate
{
	return [_date addTimeInterval:CEV_VIEWSECONDS];
}

- (void) nextDate
{
	ASSIGN(_date, [_date addTimeInterval:CEV_VIEWSECONDS]);
	[self _dateChanged];
	[self setNeedsDisplay:YES];
}

- (void) previousDate
{
	ASSIGN(_date, [_date addTimeInterval:-CEV_VIEWSECONDS]);
	[self _dateChanged];
	[self setNeedsDisplay:YES];
}

@end
