/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include "coordsys.h"

GList *coordsystems = 0;

struct coordsys equcoords =
{
	{1.0, 0.0, 0.0},
	{0.0, 1.0, 0.0},
	{0.0, 0.0, 1.0},
	0,
	12.0/PI,
	180.0/PI,
	EQUATORIALCOORDS,
	"RA.",
	"Dec.",
	"hours",
	"degrees",
	"North pole",
	"South pole",
	"NP",
	"SP"
};

struct coordsys galcoords =
{
	{-0.054876, 0.873437, -0.483835},
	{-0.494109, -0.444830, -0.746982},
	{-0.867666, 0.198076, 0.455984},
	0,
	180.0/PI,
	180.0/PI,
	GALACTICCOORDS,
	"Galactic longitude",
	"Galactic latitude",
	"degrees",
	"degrees",
	"North galactic pole",
	"South galactic pole",
	"NGP",
	"SGP"
};

struct coordsys eclcoords =
{
	{1.0, 0.0, 0.0},
	{0.0, 0.917482, -0.397777},
	{0.0, 0.397777, 0.917482},
	0,
	180.0/PI,
	180.0/PI,
	ECLIPTICCOORDS,
	"Ecliptic longitude",
	"Ecliptic latitude",
	"degrees",
	"degrees",
	"North ecliptic pole",
	"South ecliptic pole",
	"NEP",
	"SEP"
};

void addcoordinatesystem(struct coordsys *cs)
{
	coordsystems = g_list_append(coordsystems, cs);
}

void initcoordinates()
{
	addcoordinatesystem(&equcoords);
	addcoordinatesystem(&galcoords);
	addcoordinatesystem(&eclcoords);
}

struct coordsys *getcoordsys(const gchar *name)
{
	GList *c;
	struct coordsys *cs;

	for(c = coordsystems; c != 0; c = c->next)
	{
		cs = (struct coordsys *)(c->data);
		if(!cs)
		{
			fprintf(stderr, "Warning, null coordsystem data\n");
			return 0;
		}
		if(strcmp(name, cs->name) == 0) return cs;
	}

	return NULL;

}

gint coordcombofill(GtkWidget *combo)
{
	GList *c, *cl = 0;
	struct coordsys *cs;
	gint count = 0;
	
	for(c = coordsystems; c != 0; c = c->next)
	{
		cs = (struct coordsys *)(c->data);
		if(!cs)
		{
			fprintf(stderr, "coordscombofill: null coordsystem data\n");
			return 0;
		}
		cl = g_list_append(cl, g_strdup(cs->name));
		count++;
	}
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), cl);

	return count;
}

struct coordprops *newcoordprops(struct coordsys *cs)
{
	struct coordprops *cp;

	if(!cs)
	{
		fprintf(stderr, "Cannot make coordprops from null coordsys\n");
		return 0;
	}

	cp = g_new(struct coordprops, 1);

	/* now set lots of defaults */
	cp->coords = cs;
	cp->gridvis = 0;
	cp->equvis = 0;
	cp->polesvis = 1;
	cp->gridlabelmode  = 0;
	cp->equlabelmode   = 0;
	cp->poleslabelmode = 0;
	cp->gridrgb[0]  = cp->gridrgb[1]  = cp->gridrgb[2]  = 128;
	cp->equrgb[0]   = cp->equrgb[1]   = cp->equrgb[2]   = 196;
	cp->polesrgb[0] = cp->polesrgb[1] = cp->polesrgb[2] = 196;
	cp->griddensitymode = 1;
	cp->equdensitymode  = 1;
	cp->griddegrees     = 30;
	cp->gridnumber      = 6;
	cp->equdegrees      = 30;
	cp->equnumber       = 3;
	
	/* special settings for equatorial coords */
	if(strcmp(cs->name, EQUATORIALCOORDS) == 0)
	{
		cp->gridvis = 1;
		cp->equvis = 1;
		cp->polesvis = 1;
		cp->gridrgb[0]  = 200;
		cp->gridrgb[1]  = 0;
		cp->gridrgb[2]  = 0;
		cp->equrgb[0]   = 255;
		cp->equrgb[1]   = 0;
		cp->equrgb[2]   = 0;
		cp->polesrgb[0] = 255;
		cp->polesrgb[1] = 0;
		cp->polesrgb[2] = 0;
	}

	/* special settings for galactic coords */
	if(strcmp(cs->name, GALACTICCOORDS) == 0)
	{
		cp->gridvis = 0;
		cp->equvis = 1;
		cp->polesvis = 1;
		cp->gridrgb[0]  = 0;
		cp->gridrgb[1]  = 200;
		cp->gridrgb[2]  = 0;
		cp->equrgb[0]   = 0;
		cp->equrgb[1]   = 255;
		cp->equrgb[2]   = 0;
		cp->polesrgb[0] = 0;
		cp->polesrgb[1] = 255;
		cp->polesrgb[2] = 0;
	}

	/* special settings for ecliptic coords */
	if(strcmp(cs->name, ECLIPTICCOORDS) == 0)
	{
		cp->gridvis = 0;
		cp->equvis = 1;
		cp->polesvis = 1;
		cp->gridrgb[0]  = 0;
		cp->gridrgb[1]  = 0;
		cp->gridrgb[2]  = 200;
		cp->equrgb[0]   = 0;
		cp->equrgb[1]   = 0;
		cp->equrgb[2]   = 255;
		cp->polesrgb[0] = 0;
		cp->polesrgb[1] = 0;
		cp->polesrgb[2] = 255;
	}

	return cp;
}

GList *newcoordpropslist()
{
	GList *newlist = 0;
	GList *c;
	struct coordsys *cs;

	for(c = coordsystems; c != 0; c = c->next)
	{
		cs = (struct coordsys *)(c->data);
		if(!cs)
		{
			fprintf(stderr, "Warning, null coordsystem data\n");
			return 0;
		}
		newlist = g_list_append(newlist, newcoordprops(cs));
	}
	return newlist;

}

void deletecoordprops(struct coordprops *cp)
{
	if(!cp)
	{
		fprintf(stderr, "Trying to delete null coordprops\n");
		return;
	}
	g_free(cp);
}

struct coordprops *copycoordprops(struct coordprops *cp)
{
	return g_memdup(cp, sizeof(struct coordprops));
}

void getcoordprops(struct coordprops *cpa, struct viewer *v)
{
	GList *cpl;
	struct coordprops *cp;

	for(cpl = v->coordinateproperties; cpl != 0; cpl = cpl->next)
	{
		cp = (struct coordprops *)(cpl->data);
		if(!cp) fprintf(stderr, "getcoordprops : cpl->data=0\n");
		else
		memcpy(cpa++, cp, sizeof(struct coordprops));
	}
}

void setcoordprops(struct viewer *v, struct coordprops *cpa)
{
	GList *cpl;
	struct coordprops *cp;

	for(cpl = v->coordinateproperties; cpl != 0; cpl = cpl->next)
	{
		cp = (struct coordprops *)(cpl->data);
		if(!cp) fprintf(stderr, "getcoordprops : cpl->data=0\n");
		else
		memcpy(cp, cpa++, sizeof(struct coordprops));
	}
}


void coordrenderpixmap(struct coordprops *cp, struct viewer *v)
{
}
