/* ***** BEGIN LICENSE BLOCK *****
 * Version: NPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Netscape Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/NPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the NPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the NPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "plugin.h"
#include "nsIServiceManager.h"
#include "nsIMemory.h"
#include "nsISupportsUtils.h"	// this is where some useful macros defined
#include <errno.h>

#ifdef GTK_ENABLED
#include "../pixmaps/logo.xpm"
#endif

extern int errno;
extern int DEBUG;

int32 STREAMBUFSIZE = 0X0FFFFFFF;

// service manager which will give the access to all public browser services
// we will use memory service as an illustration
nsIServiceManager *gServiceManager = NULL;

// Unix needs this
#ifdef XP_UNIX

char *NPP_GetMIMEDescription(void)
{
    return GetMIMEDescription();
}

// get values per plugin
NPError NS_PluginGetValue(NPPVariable aVariable, void *aValue)
{
    return GetValue(aVariable, aValue);
}
#endif				//XP_UNIX

//////////////////////////////////////
//
// general initialization and shutdown
//
NPError NS_PluginInitialize()
{
    // this is probably a good place to get the service manager
    // note that Mozilla will add reference, so do not forget to release
    nsISupports *sm = NULL;


    NPN_GetValue(NULL, NPNVserviceManager, &sm);

    // Mozilla returns nsIServiceManager so we can use it directly; doing QI on
    // nsISupports here can still be more appropriate in case something is changed
    // in the future so we don't need to do casting of any sort.
    if (sm) {
	sm->QueryInterface(NS_GET_IID(nsIServiceManager),
			   (void **) &gServiceManager);
	NS_RELEASE(sm);
    }

    return NPERR_NO_ERROR;
}

void NS_PluginShutdown()
{
    // we should release the service manager
    NS_IF_RELEASE(gServiceManager);
    gServiceManager = NULL;
}

/////////////////////////////////////////////////////////////
//
// construction and destruction of our plugin instance object
//
nsPluginInstanceBase *NS_NewPluginInstance(nsPluginCreateData *
					   aCreateDataStruct)
{
    if (!aCreateDataStruct)
	return NULL;

    nsPluginInstance *plugin =
	new nsPluginInstance(aCreateDataStruct->instance);

    New(plugin, aCreateDataStruct);
    return plugin;
}

void NS_DestroyPluginInstance(nsPluginInstanceBase * aPlugin)
{
    if (aPlugin)
	delete(nsPluginInstance *) aPlugin;
}

////////////////////////////////////////
//
// nsPluginInstance class implementation
//
nsPluginInstance::nsPluginInstance(NPP aInstance):nsPluginInstanceBase(),
mInstance(aInstance),
mInitialized(FALSE), mScriptablePeer(NULL), mControlsScriptablePeer(NULL)
{
    int i;

    toolkitok = toolkitOk(mInstance, &moz_toolkit, &plug_toolkit);

    mScriptablePeer = getScriptablePeer();
    mControlsScriptablePeer = getControlsScriptablePeer();
    mScriptablePeer->InitControls(mControlsScriptablePeer);
    mControlsScriptablePeer->AddRef();

    // init the instance variables here.
    mimetype = NULL;
    state = 0;
    url = NULL;
    fname = NULL;
    href = NULL;
    lastmessage = (char *) NPN_MemAlloc(sizeof(char) * 1024);
    memset(lastmessage, '\0', 1);
    mode = 0;
    window_width = 0;
    window_height = 0;
    embed_width = 0;
    embed_height = 0;
    setwindow = 0;
    baseurl = NULL;
    hostname = NULL;
    pid = 0;
    noredraw = 0;
    hrefrequested = 0;
    threadsetup = 0;
    threadlaunched = 0;
    threadsignaled = 0;
    cancelled = 0;
    list = newNode();
    currentnode = NULL;
    td = (ThreadData *) NPN_MemAlloc(sizeof(ThreadData));
    td->list = NULL;
    td->instance = NULL;
    control = -1;
    player = NULL;
    autostart = 1;
    showcontrols = 1;
    mmsstream = 0;
    js_state = JS_STATE_UNDEFINED;
    nQtNext = 0;
    for (i = 0; i < 256; i++)
	qtNext[i] = NULL;
    widget = 0;
    display = NULL;
    window = 0;
    player_window = 0;
    controlwindow = 0;
    panel_height = 0;
    panel_drawn = 0;
    mediaCompleteCallback = NULL;
    mediaLength = 0.0;
    mediaPercent = 0;
    mediaTime = 0.0;
    nomediacache = 0;
    controlsvisible = 0;
    fullscreen = 0;
    showlogo = 1;
    DPMSEnabled = 0;

#ifdef GTK_ENABLED
    image = NULL;
    progress_bar = NULL;
    mediaprogress_bar = NULL;
    status = NULL;
    play_event_box = NULL;
    pause_event_box = NULL;
    stop_event_box = NULL;
    ff_event_box = NULL;
    rew_event_box = NULL;
    fs_event_box = NULL;
    gtkwidget = NULL;
    fixed_container = NULL;
    drawing_area = NULL;
    popup_menu = NULL;
#ifdef GTK2_ENABLED
    fs_window = NULL;
#endif
#endif

#ifdef X_ENABLED
    font = NULL;
    logo = (Pixmap) NULL;
    logomask = (Pixmap) NULL;
    progress_left = (Pixmap) NULL;
    progress_leftmask = (Pixmap) NULL;
    progress_middle = (Pixmap) NULL;
    progress_middlemask = (Pixmap) NULL;
    progress_right = (Pixmap) NULL;
    progress_rightmask = (Pixmap) NULL;
    progress_fill = (Pixmap) NULL;
    progress_fillmask = (Pixmap) NULL;
    lastpercent = -1;
#endif

    // options
    vo = NULL;
    vop = NULL;
    novop = 0;
    noembed = 0;
    ao = NULL;
    loop = 0;
    rtsp_use_tcp = 0;
    keep_download = 0;
    maintain_aspect = 1;
    download_dir = strdup(getenv("HOME"));
    cachesize = 512;
    output_display = NULL;
    osdlevel = 0;
    qt_speed = SPEED_MED;
    cache_percent = 25;

    // JavaScript
    paused = 0;

    // thread setup
    pthread_mutex_init(&playlist_mutex, NULL);
    pthread_mutex_init(&playlist_cond_mutex, NULL);
    pthread_mutex_init(&control_mutex, NULL);
    pthread_attr_init(&thread_attr);
    pthread_cond_init(&playlist_complete_cond, NULL);


    LoadConfigFile(this);

}

nsPluginInstance::~nsPluginInstance()
{
    // mScriptablePeer may be also held by the browser
    // so releasing it here does not guarantee that it is over
    // we should take precaution in case it will be called later
    // and zero its mPlugin member
    if (DEBUG)
	printf("~nsPluginInstance called\n");
#ifdef GTK_ENABLED
    gdk_flush();
#endif

    mInstance = NULL;
    mInitialized = FALSE;
    if (mControlsScriptablePeer != NULL) {
	mControlsScriptablePeer->SetInstance(NULL);
	mControlsScriptablePeer->Release();
	NS_IF_RELEASE(mControlsScriptablePeer);
    }

    if (mScriptablePeer != NULL) {
	mScriptablePeer->InitControls(NULL);
	mScriptablePeer->SetInstance(NULL);
	NS_IF_RELEASE(mScriptablePeer);
    }
}

NPBool nsPluginInstance::init(NPWindow * aWindow)
{
    if (aWindow == NULL)
	return FALSE;

    mInitialized = TRUE;
    return TRUE;
}

void nsPluginInstance::shut()
{
    int i;

    if (DEBUG)
	printf("shut called\n");

    if (pid != 0) {
	killmplayer(this);
    }
    if (DEBUG)
	printf("mplayer dead\n");

#ifdef X_ENABLED
    FreeUI((Display *) display, this);
#endif

    mInitialized = FALSE;

#ifdef GTK_ENABLED
    while (g_idle_remove_by_data(this)) {
	if (DEBUG)
	    printf("Removing function from idle handler\n");
    }
#endif

#ifdef GTK2_ENABLED		// When disabled probably causes a GTK memory leak, but stops weird crash and warnings
    if (controlwindow == 0) {
	if (panel_drawn == 1) {
	    gtk_widget_destroy(play_event_box);
	    gtk_widget_destroy(pause_event_box);
	    gtk_widget_destroy(stop_event_box);
	    if (mmsstream == 0) {
		gtk_widget_destroy(ff_event_box);
		gtk_widget_destroy(rew_event_box);
	    }
	    if (mediaprogress_bar != NULL)
		gtk_widget_destroy(GTK_WIDGET(mediaprogress_bar));
	    mediaprogress_bar = NULL;
	    panel_drawn = 0;
	}
	if (mode == NP_EMBED) {
	    if (player_window != 0 && noembed == 0) {
		gtk_widget_destroy(drawing_area);
		player_window = 0;
	    }
	}
	if (image != NULL)
	    gtk_widget_destroy(image);
	if (fs_event_box != NULL)
	    gtk_widget_destroy(fs_event_box);

	if (popup_menu != NULL) {
	    gtk_widget_destroy(GTK_WIDGET(menuitem_play));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_pause));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_stop));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_sep1));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_sep2));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_showcontrols));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_fullscreen));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_copy));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_save));
	    gtk_widget_destroy(GTK_WIDGET(popup_menu));
	}

	if (gtkwidget != NULL)
	    gtk_widget_destroy(gtkwidget);
    }
#endif

#ifdef GTK1_ENABLED
    if (controlwindow == 0) {

	if (popup_menu != NULL) {
	    gtk_widget_destroy(GTK_WIDGET(menuitem_play));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_pause));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_stop));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_sep1));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_sep2));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_showcontrols));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_fullscreen));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_copy));
	    gtk_widget_destroy(GTK_WIDGET(menuitem_save));
	    gtk_widget_destroy(GTK_WIDGET(popup_menu));
	}
	if (GTK_IS_WIDGET(gtkwidget))
	    gtk_widget_destroy(gtkwidget);
    }
    gdk_pixmap_unref(logo);
    gdk_pixmap_unref(pb_sm_play_up);
    gdk_pixmap_unref(pb_sm_play_down);
    gdk_pixmap_unref(pb_sm_pause_up);
    gdk_pixmap_unref(pb_sm_pause_down);
    gdk_pixmap_unref(pb_sm_stop_up);
    gdk_pixmap_unref(pb_sm_stop_down);
    gdk_pixmap_unref(pb_sm_ff_up);
    gdk_pixmap_unref(pb_sm_ff_down);
    gdk_pixmap_unref(pb_sm_rew_up);
    gdk_pixmap_unref(pb_sm_rew_down);
    gdk_pixmap_unref(pb_sm_fs_up);
    gdk_pixmap_unref(pb_sm_fs_down);

    gdk_flush();
    XSync(display, FALSE);

#endif

#ifdef GTK_ENABLED
    image = NULL;
    progress_bar = NULL;
    mediaprogress_bar = NULL;
    status = NULL;
    play_event_box = NULL;
    pause_event_box = NULL;
    stop_event_box = NULL;
    ff_event_box = NULL;
    rew_event_box = NULL;
    fs_event_box = NULL;
    gtkwidget = NULL;
    fixed_container = NULL;
    drawing_area = NULL;
    popup_menu = NULL;
#ifdef GTK2_ENABLED
    fs_window = NULL;
#endif
    gtkwidget = NULL;
#endif

    if (mimetype != NULL) {
	free(mimetype);
	mimetype = NULL;
    }

    if (href != NULL) {
	free(href);
	href = NULL;
    }

    if (fname != NULL) {
	free(fname);
	fname = NULL;
    }

    if (url != NULL) {
	free(url);
	url = NULL;
    }

    if (baseurl != NULL) {
	NPN_MemFree(baseurl);
	baseurl = NULL;
    }

    if (hostname != NULL) {
	NPN_MemFree(hostname);
	hostname = NULL;
    }

    if (vo != NULL) {
	free(vo);
	vo = NULL;
    }

    if (vop != NULL) {
	free(vop);
	vop = NULL;
    }

    if (ao != NULL) {
	free(ao);
	ao = NULL;
    }

    if (output_display != NULL) {
	free(output_display);
	output_display = NULL;
    }

    nQtNext = 0;
    for (i = 0; i < 256; i++) {
	if (qtNext[i] != NULL)
	    free(qtNext[i]);
	qtNext[i] = NULL;
    }

    if (download_dir != NULL) {
	free(download_dir);
	download_dir = NULL;
    }

    if (td->list != NULL) {
	deleteList(td->list);
	td->list = NULL;
	list = NULL;
    }

    if (td != NULL) {
	td->instance = NULL;
	NPN_MemFree(td);
	td = NULL;
    }

    if (lastmessage != NULL) {
	NPN_MemFree(lastmessage);
	lastmessage = NULL;
    }

    if (mediaCompleteCallback != NULL) {
	NPN_MemFree(mediaCompleteCallback);
	mediaCompleteCallback = NULL;
    }

    if (DEBUG)
	printf("memory free\n");

    autostart = 1;
    showcontrols = 1;
    panel_drawn = 0;
    mmsstream = 0;
    cancelled = 0;
    js_state = JS_STATE_UNDEFINED;

    pthread_attr_destroy(&thread_attr);
    pthread_mutex_destroy(&playlist_mutex);
    pthread_mutex_destroy(&playlist_cond_mutex);
    pthread_mutex_destroy(&control_mutex);
    pthread_cond_destroy(&playlist_complete_cond);

}

NPBool nsPluginInstance::isInitialized()
{
    return mInitialized;
}

NPError nsPluginInstance::SetWindow(NPWindow * aWindow)
{
    NPSetWindowCallbackStruct *ws;
    GC black_gc;
    XGCValues values;
    char message[100];
#ifdef GTK_ENABLED
    int multiplier, height, width;
#endif
#ifdef GTK1_ENABLED
//    GtkStyle *s;
#endif

    if (DEBUG >= 2)
	printf("*****SetWindow Callback Enter************\n");


    if ((aWindow == NULL) || (aWindow->window == NULL)) {
	return NPERR_NO_ERROR;
    }


    if ((Window) window != (Window) aWindow->window) {
	if (DEBUG)
	    printf("New window! old: 0x%x    new 0x%x\n",
		   (unsigned int) window, (unsigned int) aWindow->window);
    }

    if (controlwindow == 1)
	return NPERR_NO_ERROR;

    if (toolkitok != 0) {	// something is wrong
	ws = (NPSetWindowCallbackStruct *) aWindow->ws_info;
	values.foreground =
	    BlackPixel(ws->display, DefaultScreen(ws->display));
	black_gc =
	    XCreateGC(ws->display, (Window) aWindow->window, GCForeground,
		      &values);
	snprintf(message, 100,
		 "Toolkit mismatch mozilla(GTK%i), plug-in(GTK%i)",
		 moz_toolkit, plug_toolkit);
	XDrawString(ws->display, (Window) aWindow->window, black_gc, 10,
		    10, message, strlen(message));
	XFreeGC(ws->display, black_gc);
	return NPERR_NO_ERROR;
    }

    if (state < STATE_WINDOWSET) {
	if (DEBUG) {
	    printf("Size: %d %d %x\n", aWindow->x, aWindow->y,
		   (unsigned int) aWindow->window);
	    printf("Size: %dx%d \n", aWindow->width, aWindow->height);
	}

	ws = (NPSetWindowCallbackStruct *) aWindow->ws_info;

	widget =
	    XtWindowToWidget((Display *) ws->display,
			     (Window) aWindow->window);
#ifdef X_ENABLED
	XtAddEventHandler(widget, ExposureMask, FALSE,
			  (XtEventHandler) RedrawCB, this);
#endif
	display = ws->display;
#ifdef DPMSEnabled
	DPMSEnabled = DPMSIsEnabled(this);
#endif
	window = (Window) aWindow->window;
	window_width = aWindow->width;
	window_height = aWindow->height;
	state = STATE_WINDOWSET;

#ifdef GTK_ENABLED
#ifdef GTK2_ENABLED
	gtkwidget = gtk_window_new(GTK_WINDOW_POPUP);
#endif
#ifdef GTK1_ENABLED
	gdk_flush();
	XSync(display, FALSE);
	gtkwidget = gtk_plug_new(window);
	gtk_widget_realize(gtkwidget);
#endif
	gdk_flush();
	gtk_widget_set_events(gtkwidget, GDK_BUTTON_PRESS_MASK);
#ifdef GTK2_ENABLED
	gtk_widget_set_size_request(gtkwidget, aWindow->width,
				    aWindow->height);
	logo = gdk_pixbuf_new_from_xpm_data((const char **) logo_xpm);
	image = gtk_image_new_from_pixbuf(logo);
#endif

#ifdef GTK1_ENABLED
	if (DEBUG > 1)
	    printf("setting window size\n");
	gtk_widget_set_usize(gtkwidget, aWindow->width, aWindow->height);

#endif

	popup_menu = GTK_MENU(gtk_menu_new());
	menuitem_play =
	    GTK_MENU_ITEM(gtk_menu_item_new_with_label("Play"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_play));
	gtk_widget_show(GTK_WIDGET(menuitem_play));
	menuitem_pause =
	    GTK_MENU_ITEM(gtk_menu_item_new_with_label("Pause"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_pause));
	gtk_widget_show(GTK_WIDGET(menuitem_pause));
	menuitem_stop =
	    GTK_MENU_ITEM(gtk_menu_item_new_with_label("Stop"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_stop));
	gtk_widget_show(GTK_WIDGET(menuitem_stop));
#ifdef GTK2_ENABLED
	menuitem_sep1 = GTK_MENU_ITEM(gtk_separator_menu_item_new());
#endif
#ifdef GTK1_ENABLED
	menuitem_sep1 = GTK_MENU_ITEM(gtk_menu_item_new());
#endif
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_sep1));
	gtk_widget_show(GTK_WIDGET(menuitem_sep1));
	menuitem_showcontrols =
	    GTK_MENU_ITEM(gtk_check_menu_item_new_with_label
			  ("Show Controls"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_showcontrols));
	gtk_widget_show(GTK_WIDGET(menuitem_showcontrols));
	if (showcontrols)
	    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM
					   (menuitem_showcontrols), TRUE);
	menuitem_fullscreen =
	    GTK_MENU_ITEM(gtk_check_menu_item_new_with_label
			  ("Full Screen"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_fullscreen));
	gtk_widget_show(GTK_WIDGET(menuitem_fullscreen));
#ifdef GTK2_ENABLED
	menuitem_sep2 = GTK_MENU_ITEM(gtk_separator_menu_item_new());
#endif
#ifdef GTK1_ENABLED
	menuitem_sep2 = GTK_MENU_ITEM(gtk_menu_item_new());
#endif
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_sep2));
	gtk_widget_show(GTK_WIDGET(menuitem_sep2));
	menuitem_copy =
	    GTK_MENU_ITEM(gtk_menu_item_new_with_label("Copy URL"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_copy));
	gtk_widget_show(GTK_WIDGET(menuitem_copy));
	menuitem_save =
	    GTK_MENU_ITEM(gtk_menu_item_new_with_label("Save"));
	gtk_menu_append(popup_menu, GTK_WIDGET(menuitem_save));
	gtk_widget_show(GTK_WIDGET(menuitem_save));
	gtk_widget_set_sensitive(GTK_WIDGET(menuitem_save), FALSE);

#ifdef GTK2_ENABLED
	g_signal_connect_swapped(GTK_OBJECT(gtkwidget),
				 "button_press_event",
				 G_CALLBACK(popup_handler),
				 GTK_OBJECT(popup_menu));
	g_signal_connect(GTK_OBJECT(menuitem_play), "activate",
			 G_CALLBACK(menuitem_play_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_pause), "activate",
			 G_CALLBACK(menuitem_pause_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_stop), "activate",
			 G_CALLBACK(menuitem_stop_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_showcontrols), "toggled",
			 G_CALLBACK(menuitem_showcontrols_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_fullscreen), "toggled",
			 G_CALLBACK(menuitem_fullscreen_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_save), "activate",
			 G_CALLBACK(menuitem_save_callback), this);
	g_signal_connect(GTK_OBJECT(menuitem_copy), "activate",
			 G_CALLBACK(menuitem_copy_callback), this);

#endif

#ifdef GTK1_ENABLED
	gtk_signal_connect_object(GTK_OBJECT(gtkwidget),
				  "button_press_event",
				  GTK_SIGNAL_FUNC(popup_handler),
				  GTK_OBJECT(popup_menu));
	gtk_signal_connect(GTK_OBJECT(menuitem_play), "activate",
			   GTK_SIGNAL_FUNC(menuitem_play_callback), this);
	gtk_signal_connect(GTK_OBJECT(menuitem_pause), "activate",
			   GTK_SIGNAL_FUNC(menuitem_pause_callback), this);
	gtk_signal_connect(GTK_OBJECT(menuitem_stop), "activate",
			   GTK_SIGNAL_FUNC(menuitem_stop_callback), this);
	gtk_signal_connect(GTK_OBJECT(menuitem_showcontrols), "toggled",
			   GTK_SIGNAL_FUNC(menuitem_showcontrols_callback),
			   this);
	gtk_signal_connect(GTK_OBJECT(menuitem_fullscreen), "toggled",
			   GTK_SIGNAL_FUNC(menuitem_fullscreen_callback),
			   this);
	gtk_signal_connect(GTK_OBJECT(menuitem_save), "activate",
			   GTK_SIGNAL_FUNC(menuitem_save_callback), this);
	gtk_signal_connect(GTK_OBJECT(menuitem_copy), "activate",
			   GTK_SIGNAL_FUNC(menuitem_copy_callback), this);

#endif
	if (DEBUG > 1)
	    printf("menus built\n");

	fixed_container = gtk_fixed_new();
	status = GTK_LABEL(gtk_label_new("Initializing"));
	progress_bar = GTK_PROGRESS_BAR(gtk_progress_bar_new());

	if (aWindow->height > 125) {

#ifdef GTK2_ENABLED
	    gtk_fixed_put(GTK_FIXED(fixed_container), image, 10, 10);
	    gtk_widget_set_size_request(GTK_WIDGET(progress_bar),
					aWindow->width - 20, 15);
	    gtk_widget_set_size_request(GTK_WIDGET(status),
					aWindow->width - 20,
					aWindow->height - 125);
#endif
#ifdef GTK1_ENABLED
	    if (DEBUG > 1)
		printf("> 125 setup\n");
	    gtk_widget_show(gtkwidget);
//          s = gtk_widget_get_style(gtkwidget);
//          color.red = s->bg[GTK_STATE_NORMAL].red;
//          color.green = s->bg[GTK_STATE_NORMAL].green;
//          color.blue = s->bg[GTK_STATE_NORMAL].blue;
//          gdk_flush();
	    logo = gdk_pixmap_create_from_xpm_d(gtkwidget->window,
						NULL, NULL, logo_xpm);
	    image = gtk_pixmap_new(logo, NULL);
	    gtk_fixed_put(GTK_FIXED(fixed_container), GTK_WIDGET(image),
			  10, 10);
	    gtk_widget_set_usize(GTK_WIDGET(progress_bar),
				 aWindow->width - 20, 15);
	    gtk_widget_set_usize(GTK_WIDGET(status),
				 aWindow->width - 20,
				 aWindow->height - 125);
#endif
	    gtk_fixed_put(GTK_FIXED(fixed_container),
			  GTK_WIDGET(progress_bar), 10,
			  aWindow->height - 50);
	    if (showlogo)
		gtk_widget_show(image);
	    gtk_widget_show(GTK_WIDGET(progress_bar));
	    gtk_fixed_put(GTK_FIXED(fixed_container), GTK_WIDGET(status),
			  10, 60);
	} else {
	    gtk_fixed_put(GTK_FIXED(fixed_container), GTK_WIDGET(status),
			  10, 10);
	    gtk_fixed_put(GTK_FIXED(fixed_container),
			  GTK_WIDGET(progress_bar), 10,
			  aWindow->height - 20);
	    gtk_widget_show(GTK_WIDGET(progress_bar));
#ifdef GTK2_ENABLED
	    gtk_widget_set_size_request(GTK_WIDGET(progress_bar),
					aWindow->width - 20, 15);
	    gtk_widget_set_size_request(GTK_WIDGET(status),
					aWindow->width - 20,
					aWindow->height - 25);
#endif
#ifdef GTK1_ENABLED
	    if (DEBUG > 1)
		printf("< 125 setup\n");
	    gtk_widget_set_usize(GTK_WIDGET(progress_bar),
				 aWindow->width - 20, 15);
	    gtk_widget_set_usize(GTK_WIDGET(status),
				 aWindow->width - 20,
				 aWindow->height - 25);
#endif
	}

	gtk_misc_set_alignment(GTK_MISC(status), 0, 0);
	gtk_label_set_line_wrap(status, TRUE);
	gtk_container_add(GTK_CONTAINER(gtkwidget), fixed_container);
	gtk_widget_show(GTK_WIDGET(status));
	gtk_widget_show(fixed_container);
	gtk_widget_show(gtkwidget);
	gdk_flush();
	InitPixbufs(this);
	if (DEBUG > 1)
	    printf("all setup and ready to reparent and map\n");
#ifdef GTK2_ENABLED
	XReparentWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			GDK_WINDOW_XWINDOW(gtkwidget->window), window, 0,
			0);
	gtk_widget_map(gtkwidget);
#endif
	if (DEBUG > 1)
	    printf("reparent and map done\n");

#endif

    } else {
	if (setwindow == 0) {
	    if (DEBUG)
		printf("back in SetWindow\n");
	    if (DEBUG)
		printf("New Size: %ix%i\n", aWindow->width,
		       aWindow->height);
	    setwindow++;

	    if (state != STATE_GETTING_PLAYLIST) {
		state = STATE_GETTING_PLAYLIST;
		if (href != NULL) {
		    if (isMms(href)) {
			snprintf(list->url, 1024, "%s", href);
			state = STATE_STARTED_PLAYER;
			list->mmsstream = 1;
			if (threadsetup == 0 && controlwindow == 0)
			    SetupPlayer(this, NULL);
		    }
		} else {
		    if (url != NULL) {
			if (isMms(url)) {
			    snprintf(list->url, 1024, "%s", url);
			    state = STATE_STARTED_PLAYER;
			    list->mmsstream = 1;
			    if (threadsetup == 0 && controlwindow == 0)
				SetupPlayer(this, NULL);
			}
		    }
		}
	    }
	}
	if (mode == NP_EMBED) {
	    embed_width = aWindow->width;
	    embed_height = aWindow->height;
	} else {
	    window_width = aWindow->width;
	    window_height = aWindow->height;
	}
    }

    if (DEBUG)
	printf("resizing widgets to %i x %i \n", aWindow->width,
	       aWindow->height);

#ifdef GTK_ENABLED
    if (status != NULL) {
	gtk_widget_show(GTK_WIDGET(status));
#ifdef GTK2_ENABLED
	gtk_widget_set_size_request(GTK_WIDGET(status),
				    aWindow->width - 20,
				    aWindow->height - 25);
#endif
#ifdef GTK1_ENABLED
	gtk_widget_set_usize(GTK_WIDGET(status),
			     aWindow->width - 20, aWindow->height - 25);
#endif
    }
    if (gtkwidget == NULL)
	return NPERR_NO_ERROR;

    gtk_widget_show(gtkwidget);
    XResizeWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
		  GDK_WINDOW_XWINDOW(gtkwidget->window),
		  aWindow->width, aWindow->height);
    XResizeWindow(display, window, aWindow->width, aWindow->height);

    if (DEBUG > 1)
	printf("resize is complete\n");

    if (panel_drawn == 1 && mode == NP_FULL) {
	if (panel_height >= 32) {
	    height = 32;
	    width = 43;
	} else {
	    height = 16;
	    width = 21;
	}

	multiplier = 0;

	if (mmsstream == 0) {
	    gtk_fixed_move(GTK_FIXED(fixed_container),
			   GTK_WIDGET(rew_event_box), width * multiplier++,
			   aWindow->height - height);
	}
	gtk_fixed_move(GTK_FIXED(fixed_container),
		       GTK_WIDGET(play_event_box), width * multiplier++,
		       aWindow->height - height);
	gtk_fixed_move(GTK_FIXED(fixed_container),
		       GTK_WIDGET(pause_event_box), width * multiplier++,
		       aWindow->height - height);
	gtk_fixed_move(GTK_FIXED(fixed_container),
		       GTK_WIDGET(stop_event_box), width * multiplier++,
		       aWindow->height - height);
	if (mmsstream == 0) {
	    gtk_fixed_move(GTK_FIXED(fixed_container),
			   GTK_WIDGET(ff_event_box), width * multiplier++,
			   aWindow->height - height);
	}
	if (mediaprogress_bar != NULL) {
	    gtk_fixed_move(GTK_FIXED(fixed_container),
			   GTK_WIDGET(mediaprogress_bar),
			   (width * multiplier + 10),
			   aWindow->height - height + 2);
	    gtk_widget_set_usize(GTK_WIDGET(mediaprogress_bar),
				 aWindow->width -
				 (width * (multiplier + 1) + 20),
				 height - 4);
	}
	gtk_fixed_move(GTK_FIXED(fixed_container),
		       GTK_WIDGET(fs_event_box),
		       (aWindow->width - width), aWindow->height - height);

    }
#endif

    if (DEBUG >= 2) {
	printf("***********SetWindow Callback Exit**************\n");
    }


    return NPERR_NO_ERROR;
}

NPError nsPluginInstance::NewStream(NPMIMEType type, NPStream * stream,
				    NPBool seekable, uint16 * stype)
{

    if (DEBUG >= 2)
	printf("**********NewStream Callback %s ****************\n",
	       stream->url);

    if (baseurl == NULL)
	baseurl = getURLBase((char *) stream->url);
    if (hostname == NULL)
	hostname = getURLHostname((char *) stream->url);

    if (mode == NP_FULL) {
	url = strdup(stream->url);
    }

    if ((threadsetup == 0) && (controlwindow == 0)) {
	state = STATE_GETTING_PLAYLIST;
	SetupPlayer(this, NULL);
    }
    *stype = NP_NORMAL;

    if (DEBUG >= 2)
	printf("*********Exiting NewStream Callback*****************\n");

    return NPERR_NO_ERROR;
}


NPError nsPluginInstance::DestroyStream(NPStream * stream, NPError reason)
{
    int playable, all_retrieved, all_above_cache;
    Node *n;

    if (DEBUG >= 2)
	printf("***********NPP_DestroyStream called %i\n URL: %s\n",
	       reason, stream->url);

    if (reason == NPRES_DONE) {
	playable = 0;
	if (strlen(stream->url) >= 1023) {
	    return NPERR_NO_ERROR;
	}
	pthread_mutex_lock(&playlist_mutex);
	n = td->list;
	while (n != NULL) {
	    if (URLcmp(n->url, stream->url) == 0) {
		if (DEBUG)
		    printf("Destroy stream found a URL match\n%s\n%s\n",
			   n->url, stream->url);
		break;
	    } else {
		if (strstr(stream->url, n->url) != NULL) {
		    break;
		}
	    }
	    n = n->next;
	}

	if (n != NULL) {
	    n->retrieved = 1;

	    if (n->localcache != NULL) {
		if (fclose(n->localcache) != 0) {
		    if (DEBUG)
			printf("fclose had an error %i : %s\n", errno,
			       strerror(errno));
		};
		n->localcache = NULL;
	    }

	    if (controlwindow == 1) {
		if (n->fname != NULL)
		    remove(n->fname);
	    }

	    if (!isMms(n->url)) {
		if (DEBUG)
		    printf("calling buildPlaylist with filename %s\n",
			   n->fname);
		buildPlaylist(this, n->fname, n);
		if (mode == NP_FULL || noembed == 1) {
#ifdef X_ENABLED
		    DrawUI(widget, this, "Download Complete", 0, 99);
#endif
#ifdef GTK_ENABLED
		    if (status != NULL) {
			gtk_label_set_text(status, "Download Complete");
		    }
		    if (progress_bar != NULL) {
			gtk_progress_bar_update(progress_bar, 1.0);
			if (n->next == NULL) {
			    gtk_widget_hide(GTK_WIDGET(progress_bar));
			} else {
			    gtk_widget_show(GTK_WIDGET(progress_bar));
			}
		    }
#endif
		}
		if (n->playlist == 1 || n->cancelled == 1) {
		    if (n->mmsstream == 0) {
			if (n->next == NULL) {
			    pthread_mutex_unlock(&playlist_mutex);
			    return NPERR_NO_ERROR;
			}
		    }
		}
	    } else {
		n->mmsstream = 1;
	    }


	    // test for all retrieved
	    if (threadsignaled == 0) {
		all_retrieved = 1;
		n = td->list;
		while (n != NULL) {
		    if ((n->retrieved == 0) && (n->play == 1)) {
			all_retrieved = 0;
			break;
		    }
		    n = n->next;
		}
		// if all retrieved signal player
		if (all_retrieved) {
		    if (threadsignaled == 0) {
			if (DEBUG)
			    printf("signalling player (retrieved)\n");
			if (autostart) {
			    signalPlayerThread(this);
			    threadsignaled = 1;
			}
		    }
		}
	    }
	    // if not then see if all not retrieved are above cache,
	    if (threadsignaled == 0) {
		all_above_cache = 1;
		n = td->list;
		while (n != NULL) {

		    if (DEBUG) {
			printf
			    ("n->url= %s\nn->bytes = %li\nn->cachebytes = %li\nn->play= %i\nn->playlist= %i\nn->mmsstream= %i\n",
			     n->url, n->bytes, n->cachebytes, n->play,
			     n->playlist, n->mmsstream);

		    }

		    if ((n->bytes <= n->cachebytes) && (n->play == 1)) {
			all_above_cache = 0;
			break;
		    }
		    n = n->next;
		}
		// if all above cache signal player
		if (all_above_cache) {
		    if (threadsignaled == 0) {
			if (DEBUG)
			    printf("signalling player (above cache)\n");
			if (autostart) {
			    signalPlayerThread(this);
			    threadsignaled = 1;
			}
		    }
		}
	    }
	    // check for streaming media
	    if (threadsignaled == 0) {
		// look for mmsstream
		n = td->list;
		while (n != NULL) {
		    if ((n->mmsstream == 1) && (n->play == 1)) {
			if (threadsignaled == 0) {
			    if (DEBUG)
				printf("signalling player (mmsstream)\n");
			    if (autostart) {
				signalPlayerThread(this);
				threadsignaled = 1;
			    } else {
				if (showcontrols) {
				    panel_height = 16;
#ifdef GTK_ENABLED
				    g_idle_add(gtkgui_draw, this);
#endif
				}
			    }
			}
			break;
		    }
		    n = n->next;
		}
	    }
	}
	pthread_mutex_unlock(&playlist_mutex);
    }

    if (DEBUG >= 2)
	printf
	    ("*******Exiting DestroyStream Callback, state = %d, js_state = %d\n",
	     state, js_state);


    return NPERR_NO_ERROR;
}

int32 nsPluginInstance::WriteReady(NPStream * stream)
{
    Node *n;

    if (state == STATE_PLAY_CANCELLED)
	return -1;

    if (td == NULL)
	return -1;

    if (strlen(stream->url) >= 1023)
	return -1;

    pthread_mutex_lock(&playlist_mutex);

    if (DEBUG >= 3)
	printf("**WriteReady for %s, state =%d, js_state = %d\n",
	       stream->url, state, js_state);

    n = td->list;
    while (n != NULL) {
	if (0)
	    printf("WR:\nn->url= %s\nstream->url= %s\n", n->url,
		   stream->url);
	if (strlen(n->url) != 0) {
	    if (URLcmp(n->url, stream->url) == 0) {
		break;
	    } else {
		if (strstr(stream->url, n->url) != NULL) {
		    break;
		}
	    }
	} else {
	    snprintf(n->url, 1024, "%s", stream->url);
	    break;
	}
	n = n->next;
    }

    if (n != NULL) {
	// if we have determined that we are not going to play the file
	// stop downloading it, and clean it up during cleanup
	if (n->cancelled == 1) {
	    n->remove = 1;
	    //check locking here
	    NPN_DestroyStream(mInstance, stream, NPRES_DONE);
	}
	// don't download it twice
	if (n->retrieved == 1) {
	    NPN_DestroyStream(mInstance, stream, NPRES_DONE);
	}
	if ((nomediacache == 1) && (stream->end > 16384)) {
	    pthread_mutex_unlock(&playlist_mutex);
	    if (threadsignaled == 0) {
		if (DEBUG)
		    printf("signalling player from write ready\n");
		if (autostart) {
		    signalPlayerThread(this);
		    threadsignaled = 1;
		}
	    }
	    return -1;

	} else {
	    if (strlen(n->fname) == 0) {
		snprintf(n->fname, 1024, "%s",
			 tempnam("/tmp", "mplayerplug-inXXXXXX"));
		if (DEBUG)
		    printf("WR tempname: %s\n", n->fname);

	    }
	    if (n->totalbytes != (int) stream->end)
		n->totalbytes = stream->end;

	    if (n->cachebytes <
		(long int) (stream->end * cache_percent / 100))
		n->cachebytes =
		    (long int) (stream->end * cache_percent / 100);

	    if (n->cachebytes < (cachesize * 1024))
		n->cachebytes = cachesize * 1024;

	    pthread_mutex_unlock(&playlist_mutex);
	    return STREAMBUFSIZE;
	}
    } else {
	if (DEBUG)
	    printf("didn't find the node in the playlist\n %s\n",
		   stream->url);
	n = newNode();
	snprintf(n->url, 1024, "%s", stream->url);
	if ((nomediacache == 1) && (stream->end > 16384)) {
	    addToEnd(td->list, n);
	    pthread_mutex_unlock(&playlist_mutex);
	    return -1;
	} else {
	    snprintf(n->fname, 1024, "%s",
		     tempnam("/tmp", "mplayerplug-inXXXXXX"));
	    addToEnd(td->list, n);
	    if (n->totalbytes != (int) stream->end)
		n->totalbytes = stream->end;
	    pthread_mutex_unlock(&playlist_mutex);
	    if (DEBUG >= 3) {
		printf
		    ("**Exiting WriteReady Callback, state = %d, js_state = %d\n",
		     state, js_state);
	    }
	    return STREAMBUFSIZE;
	}
    }

}

int32 nsPluginInstance::Write(NPStream * stream, int32 offset, int32 len,
			      void *buffer)
{
    int ret = 0;
    long int currdownload, maxdownload;
    FILE *fp;
    Node *n, *hrefnode;
    char message[1024];
    char *burl;

    if (state == STATE_PLAY_CANCELLED)
	return -1;
    if (td == NULL)
	return -1;

    if (DEBUG >= 3)
	printf("****Write Callback %s : %i : %i\n", stream->url,
	       offset, len);

    if (strlen(stream->url) >= 1023)
	return -1;

    pthread_mutex_lock(&playlist_mutex);

    // find the node that matches the URL and open the file name in it.
    currdownload = 0;
    maxdownload = 0;
    for (n = td->list; n != NULL; n = n->next) {
	//if (URLcmp(n->url, stream->url) == 0) continue;
	//if (strstr(stream->url, n->url) == NULL) continue;
	if (n->play == 0)
	    continue;
	if (n->cancelled)
	    continue;
	currdownload += n->bytes;
	maxdownload += n->totalbytes;
    }

    n = td->list;
    while (n != NULL) {
	if (URLcmp(n->url, stream->url) == 0) {
	    break;
	} else {
	    if (strstr(stream->url, n->url) != NULL) {
		break;
	    }
	}
	n = n->next;
    }

    if (n == NULL) {
	pthread_mutex_unlock(&playlist_mutex);
	return -1;
    } else {
	if (n->cancelled == 1 || state == STATE_PLAY_CANCELLED) {
	    if (n->localcache != NULL) {
		fclose(n->localcache);
		n->localcache = NULL;
	    }
	    pthread_mutex_unlock(&playlist_mutex);

	    if (DEBUG) {
		printf
		    ("*******Exiting Write: CANCELLED, state = %d, js_state = %d\n",
		     state, js_state);
	    }

	    return -1;
	}

	if (n->status != STATE_CANCELLED) {

	    if (n->localcache == NULL) {
		fp = fopen(n->fname, "a+");
		n->localcache = fp;
	    } else {
		fp = n->localcache;
	    }
	    if (fp == NULL) {
		pthread_mutex_unlock(&playlist_mutex);
		return -1;
	    }
	    fseek(fp, offset, SEEK_SET);
	    ret = fwrite(buffer, 1, len, fp);
	    n->bytes = n->bytes + ret;
	    if (maxdownload == 0) {
		snprintf(message, 1024, "Buffering %li KB ",
			 (n->bytes / 1024));

	    } else {
		snprintf(message, 1024, "Buffering %i%% - %li KB ",
			 (int) ((unsigned long) (currdownload * 100) /
				maxdownload), (n->bytes / 1024));
	    }
	    if (mode == NP_EMBED && noembed == 0 && fullscreen == 0) {
		if (state < STATE_PLAYING) {
#ifdef X_ENABLED
		    DrawUI(widget, this, message,
			   0,
			   (int) ((currdownload / (maxdownload * 1.0)) *
				  100));
#endif
#ifdef GTK_ENABLED
		    if (progress_bar != NULL && maxdownload != 0) {
			gtk_progress_bar_update(progress_bar,
						((currdownload * 1.0) /
						 (maxdownload * 1.0)));
			gtk_widget_show(GTK_WIDGET(progress_bar));
			gtk_widget_queue_draw(GTK_WIDGET(progress_bar));
		    }
		    if (status != NULL) {
			gtk_label_set_text(status, message);
			gtk_widget_show(GTK_WIDGET(status));
			gtk_widget_queue_draw(GTK_WIDGET(status));
		    }
#endif
		}
	    } else {
		if (fullscreen == 0) {
#ifdef X_ENABLED
		    DrawUI(widget, this, message,
			   0,
			   (int) ((currdownload / (maxdownload * 1.0)) *
				  100));
#endif
#ifdef GTK_ENABLED
		    if (progress_bar != NULL && maxdownload != 0) {
			gtk_progress_bar_update(progress_bar,
						((currdownload * 1.0) /
						 (maxdownload * 1.0)));
			gtk_widget_show(GTK_WIDGET(progress_bar));
			gtk_widget_queue_draw(GTK_WIDGET(progress_bar));
		    }
		    if (status != NULL) {
			if (state < STATE_PLAYING) {
			    gtk_label_set_text(status, message);
			    gtk_widget_show(GTK_WIDGET(status));
			    gtk_widget_queue_draw(GTK_WIDGET(status));
			}
		    }
#endif
		}
	    }

	    if (href != NULL) {
		if (hrefrequested == 0) {
		    if (DEBUG)
			printf("requesting href\n");
		    hrefrequested = 1;
		    n = td->list;
		    while (n != NULL) {
			if (URLcmp(n->url, href) == 0) {
			    break;
			} else {
			    if (strstr(stream->url, href) != NULL) {
				break;
			    }
			}
			n = n->next;
		    }
		    if (n == NULL) {
			if (DEBUG)
			    printf("href not in list\n");
			hrefnode = newNode();
			strlcpy(hrefnode->url, href,
				sizeof(hrefnode->url));
			addToEnd(td->list, hrefnode);
		    } else {
			if (DEBUG)
			    printf("href in list\n");
		    }
		    NPN_GetURL(mInstance, href, NULL);
		    pthread_mutex_unlock(&playlist_mutex);

		    if (DEBUG) {
			printf
			    ("*******Exiting Write, state = %d, js_state = %d\n",
			     state, js_state);
		    }

		    return ret;
		}
	    }

	    if (n->status != STATE_DOWNLOADED_ENOUGH) {
		burl = getURLBase(n->url);
		if (burl != NULL) {
		    if (baseurl == NULL) {
			baseurl = burl;
		    } else {
			if (strcmp(baseurl, burl) != 0) {
			    NPN_MemFree(baseurl);
			    baseurl = burl;
			} else {
			    NPN_MemFree(burl);
			}
		    }
		}

		if (isMms(n->url)) {
		    n->mmsstream = 1;
		}
	    }
	    if (n->play == 1) {
		if (n->mmsstream == 1) {
		    if (threadsignaled == 0) {
			if (DEBUG)
			    printf("signalling player from write\n");
			if (autostart) {
			    signalPlayerThread(this);
			    threadsignaled = 1;
			}
		    }
		} else {
		    //if (This->mode != NP_FULL) {
		    if (n->bytes > n->cachebytes) {
			if (threadsignaled == 0) {
			    if (DEBUG)
				printf("signalling player from write\n");
			    if (autostart) {
				signalPlayerThread(this);
				threadsignaled = 1;
			    } else {
				if (showcontrols) {
				    panel_height = 16;
#ifdef GTK_ENABLED
				    g_idle_add(gtkgui_draw, this);
#endif
				}
			    }
			}
		    }
		    //}
		}
	    }

	}
	n->status = STATE_DOWNLOADED_ENOUGH;
	pthread_mutex_unlock(&playlist_mutex);

	if (DEBUG >= 3) {
	    printf("*******Exiting Write, state = %d, js_state = %d\n",
		   state, js_state);
	}
	return ret;
    }


}

// methods called from nsScriptablePeer

void nsPluginInstance::Play()
{
    Node *n;

    if (DEBUG > 1) {
	printf("*****Play Called\n");
    }

    pthread_mutex_lock(&control_mutex);
    if (paused == 1) {
	if (DEBUG)
	    printf("sending play\n");
	sendCommand(this, "pause\n");
	paused = 0;
	js_state = JS_STATE_PLAYING;
    }

    if (js_state == JS_STATE_UNDEFINED) {
	//reset the playlist
	if (DEBUG > 1)
	    printf("Play: resetting playlist\n");

	pthread_mutex_lock(&playlist_mutex);
	n = list;
	while (n != NULL) {
	    if (n->play)
		n->played = 0;
	    n = n->next;
	}
	pthread_mutex_unlock(&playlist_mutex);
    }

    if (threadsetup == 0 && controlwindow == 0) {
	if (DEBUG > 1)
	    printf("Play: setupplayer\n");
	state = STATE_GETTING_PLAYLIST;
	pthread_mutex_unlock(&control_mutex);
	SetupPlayer(this, NULL);
	pthread_mutex_lock(&control_mutex);
    }

    if (threadsignaled == 1 && js_state == JS_STATE_UNDEFINED) {
	if (DEBUG > 1)
	    printf("Play: launching thread\n");
	state = STATE_NEWINSTANCE;
	launchPlayerThread(this);
	pthread_mutex_unlock(&control_mutex);
	// recommended slight pause
	usleep(1);
	//signal player thread
	if (DEBUG > 1)
	    printf("Play: signaling thread after launch\n");
	signalPlayerThread(this);
	threadsignaled = 1;

    } else if ((autostart == 0) && (threadsignaled == 0)) {
	pthread_mutex_unlock(&control_mutex);
	if (DEBUG > 1)
	    printf("Play: signaling thread without launch\n");

	signalPlayerThread(this);
	threadsignaled = 1;
    } else {
	pthread_mutex_unlock(&control_mutex);
    }

#ifdef GTK_ENABLED
    play_callback(NULL, NULL, this);
#endif

    if (DEBUG > 1) {
	printf("***********Exiting Play*************\n");
    }

}

void nsPluginInstance::Pause()
{
    pthread_mutex_lock(&control_mutex);
    if (paused == 0) {
	if (DEBUG)
	    printf("sending pause\n");
	sendCommand(this, "pause\n");
#ifdef GTK_ENABLED
	pause_callback(NULL, NULL, this);
#endif
	paused = 1;
	js_state = JS_STATE_PAUSED;
    }
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::Stop()
{
    pthread_mutex_lock(&control_mutex);
    if (DEBUG)
	printf("sending stop\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    sendCommand(this, "seek 0 2\npause\n");
#ifdef GTK_ENABLED
    stop_callback(NULL, NULL, this);
#endif
    paused = 1;
    js_state = JS_STATE_STOPPED;
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::Quit()
{
    pthread_mutex_lock(&control_mutex);
    if (DEBUG)
	printf("sending quit\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    sendCommand(this, "quit\n");
    paused = 0;
    threadsetup = 0;
    threadsignaled = 0;
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::FastForward()
{
    pthread_mutex_lock(&control_mutex);
    js_state = JS_STATE_SCANFORWARD;
    if (DEBUG)
	printf("sending FastForward\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    sendCommand(this, "seek +10 0\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::FastReverse()
{

    pthread_mutex_lock(&control_mutex);
    js_state = JS_STATE_SCANREVERSE;
    if (DEBUG)
	printf("sending FastReverse\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    sendCommand(this, "seek -10 0\n");
    if (paused == 1)
	sendCommand(this, "pause\n");
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::Seek(double counter)
{
    char command[32];

    pthread_mutex_lock(&control_mutex);
    if (paused == 1)
	sendCommand(this, "pause\n");
    snprintf(command, 32, "seek %5.0f 2\n", counter);
    sendCommand(this, command);
    if (paused == 1)
	sendCommand(this, "pause\n");
    pthread_mutex_unlock(&control_mutex);

}

void nsPluginInstance::GetPlayState(PRInt32 * playstate)
{
    pthread_mutex_lock(&control_mutex);
    *playstate = js_state;
    pthread_mutex_unlock(&control_mutex);
}

void nsPluginInstance::GetTime(double *_retval)
{
    *_retval = (double) mediaTime;
}

void nsPluginInstance::GetDuration(double *_retval)
{
    *_retval = (double) mediaLength;
}

void nsPluginInstance::GetPercent(double *_retval)
{
    *_retval = (double) mediaPercent;
}

void nsPluginInstance::GetFilename(char **filename)
{
    if (DEBUG >= 2)
	printf("***************** GetFilename called %s\n", *filename);
    if (href != NULL)
	*filename = strdup(href);
    if (fname != NULL)
	*filename = strdup(fname);
    if (url != NULL)
	*filename = strdup(url);
    if (DEBUG >= 2)
	printf("***************** GetFilename exited %s\n", *filename);
}

void nsPluginInstance::SetFilename(const char *filename)
{
    char localurl[1024];

    if (DEBUG >= 2)
	printf("***************** SetFilename called %s\n", filename);
    killmplayer(this);
    // reset some vars
    paused = 0;
    threadsetup = 0;
    threadsignaled = 0;
    // reset the list
    pthread_mutex_lock(&playlist_mutex);
    deleteList(list);
    list = newNode();
    td->list = NULL;

    // need to convert to Fully Qualified URL here
    fullyQualifyURL(this, (char *) filename, localurl);

    if (href != NULL) {
	free(href);
	href = NULL;
    }

    if (fname != NULL) {
	free(fname);
	fname = NULL;
    }

    if (url != NULL) {
	free(url);
	url = NULL;
    }

    url = strdup(localurl);
    cancelled = 0;
    if (DEBUG)
	printf("SetFilename getting %s\n", localurl);
    if (!isMms(localurl))
	NPN_GetURL(mInstance, localurl, NULL);

    pthread_mutex_unlock(&playlist_mutex);

    if (DEBUG >= 2) {
	printf("**********SetFilename Exit***************\n");
    }
}

void nsPluginInstance::GetShowControls(PRBool * _retval)
{
    *_retval = (PRBool) controlsvisible;

}

void nsPluginInstance::SetShowControls(PRBool value)
{
#ifdef GTK_ENABLED
    if (value) {
	if (panel_drawn == 0) {
	    gtkgui_draw(this);
	} else {
	    if (play_event_box != NULL)
		gtk_widget_show(play_event_box);
	    if (pause_event_box != NULL)
		gtk_widget_show(pause_event_box);
	    if (stop_event_box != NULL)
		gtk_widget_show(stop_event_box);
	    if (ff_event_box != NULL)
		gtk_widget_show(ff_event_box);
	    if (rew_event_box != NULL)
		gtk_widget_show(rew_event_box);
	    if (mediaprogress_bar != NULL && mediaPercent > 0)
		gtk_widget_show(GTK_WIDGET(mediaprogress_bar));
	    if (fs_event_box != NULL)
		gtk_widget_show(GTK_WIDGET(fs_event_box));
	}
	controlsvisible = 1;

    } else {			// hide everything

	if (panel_drawn == 0) {
	    // do nothing
	} else {
	    if (play_event_box != NULL)
		gtk_widget_hide(play_event_box);
	    if (pause_event_box != NULL)
		gtk_widget_hide(pause_event_box);
	    if (stop_event_box != NULL)
		gtk_widget_hide(stop_event_box);
	    if (ff_event_box != NULL)
		gtk_widget_hide(ff_event_box);
	    if (rew_event_box != NULL)
		gtk_widget_hide(rew_event_box);
	    if (mediaprogress_bar != NULL)
		gtk_widget_hide(GTK_WIDGET(mediaprogress_bar));
	    if (fs_event_box != NULL)
		gtk_widget_hide(GTK_WIDGET(fs_event_box));
	}
	controlsvisible = 0;
    }

#endif
#ifdef GTK2_ENABLED
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM
				   (menuitem_showcontrols),
				   controlsvisible);
#endif
}

void nsPluginInstance::GetFullscreen(PRBool * _retval)
{
    *_retval = (PRBool) fullscreen;

}

void nsPluginInstance::SetFullscreen(PRBool value)
{

#ifdef GTK2_ENABLED
    GdkDisplay *gdisp;
    GdkScreen *screen;
    GdkColor black;
    GdkRectangle rect;
#endif
#ifdef GTK1_ENABLED
    GdkColor black;
#endif

#ifdef GTK_ENABLED
    int multiplier, height = 0, width;
    int x, y, disp_x, pos_x, disp_y, pos_y;
#endif
    int win_height, win_width;

    if (DEBUG)
	printf("in SetFullscreen with %i\nfullscreen = %i\n", value,
	       fullscreen);

#ifdef GTK_ENABLED
    black.red = 0;
    black.green = 0;
    black.blue = 0;
#endif
    if (mode == NP_EMBED) {
	win_height = embed_height;
	win_width = embed_width;
    } else {
	win_height = window_height;
	win_width = window_width;
    }

    if (DEBUG)
	printf("height = %i and width = %i\n", win_height, win_width);

    if (win_height == 0 || win_width == 0 || hidden == 1)
	return;

    if (fullscreen) {
	if (value) {
	    // do nothing
	    if (DEBUG > 1)
		printf("SetFullscreen doing nothing\n");
	    fullscreen = 1;
	} else {
#ifdef GTK_ENABLED
	    if (DEBUG > 1)
		printf("SetFullscreen returning to original size\n");
#ifdef GTK2_ENABLED
	    if (fs_window != NULL) {
		gtk_window_unfullscreen(GTK_WINDOW(fs_window));
		XReparentWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
				GDK_WINDOW_XWINDOW(gtkwidget->window),
				window, 0, 0);
		gtk_widget_map(gtkwidget);
		gtk_widget_destroy(fs_window);
		fs_window = NULL;
	    }
	    gtk_window_move(GTK_WINDOW(gtkwidget), 0, 0);
	    gtk_window_resize(GTK_WINDOW(gtkwidget), win_width,
			      win_height);
#endif
#ifdef GTK1_ENABLED
	    if (fs_window != NULL) {
		XReparentWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
				GDK_WINDOW_XWINDOW(gtkwidget->window),
				window, 0, 0);
		XMapWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			   GDK_WINDOW_XWINDOW(gtkwidget->window));
		gtk_widget_destroy(fs_window);
		fs_window = NULL;
	    }

	    XResizeWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			  GDK_WINDOW_XWINDOW(gtkwidget->window),
			  win_width, win_height);
#endif
	    if (panel_drawn == 1) {
		height = 16;
		width = 21;

		multiplier = 0;

		if (mmsstream == 0) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(rew_event_box),
				   width * multiplier++,
				   win_height - height);
		}
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(play_event_box),
			       width * multiplier++, win_height - height);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(pause_event_box),
			       width * multiplier++, win_height - height);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(stop_event_box),
			       width * multiplier++, win_height - height);
		if (mmsstream == 0) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(ff_event_box),
				   width * multiplier++,
				   win_height - height);
		}

		if (mediaprogress_bar != NULL) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(mediaprogress_bar),
				   (width * multiplier + 10),
				   win_height - height + 2);
		    gtk_widget_set_usize(GTK_WIDGET(mediaprogress_bar),
					 win_width -
					 (width * (multiplier + 1) + 20),
					 height - 4);
		}
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(fs_event_box),
			       (win_width - width), win_height - height);


	    }

	    if (image != NULL && showlogo)
		gtk_widget_show(GTK_WIDGET(image));

	    if (embed_height > 125 || mode == NP_FULL) {
		gtk_widget_set_usize(GTK_WIDGET(status), embed_width - 20,
				     embed_height - 125);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(status), 10, 60);
	    } else {
		gtk_widget_set_usize(GTK_WIDGET(status), embed_width - 20,
				     embed_height - 20);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(status), 10, 10);
	    }

	    if (mode == NP_EMBED) {
		if (movie_height != 0 && movie_width != 0) {
		    if (drawing_area != NULL)
			gtk_widget_set_usize(drawing_area,
					     movie_width, movie_height);
		} else {
		    if (drawing_area != NULL)
			gtk_widget_set_usize(drawing_area,
					     embed_width, embed_height);
		}
		if (drawing_area != NULL)
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(drawing_area), 0, 0);
	    } else {
		if (drawing_area != NULL) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(drawing_area), 10, 100);
		    if (movie_height != 0 && movie_width != 0) {
			gtk_widget_set_usize(drawing_area,
					     movie_width, movie_height);
			if ((movie_width + 10 >= window_width)
			    || (movie_height + 100 >= window_height))
			    gtk_fixed_put(GTK_FIXED(fixed_container),
					  drawing_area, 0, 0);
		    } else {
			gtk_widget_set_usize(drawing_area,
					     window_width, window_height);
		    }
		}
	    }
#ifdef GTK2_ENABLED
	    gtk_widget_modify_bg(gtkwidget, GTK_STATE_NORMAL,
				 &(gtk_widget_get_style(image)->
				   bg[GTK_STATE_NORMAL]));
	    if (drawing_area != NULL)
		gtk_widget_modify_bg(drawing_area, GTK_STATE_NORMAL,
				     &(gtk_widget_get_style(image)->
				       bg[GTK_STATE_NORMAL]));
#endif
#ifdef GTK1_ENABLED
// This style block does not seem to work anyway and since color is not set it could crash
//          style = gtk_rc_style_new();
//          style->bg[GTK_STATE_NORMAL] = color;
//          style->color_flags[GTK_STATE_NORMAL] =
//              (GtkRcFlags) (style->
//                            color_flags[GTK_STATE_NORMAL] | GTK_RC_BG);
//          gtk_widget_modify_style(gtkwidget, style);
//          if (drawing_area != NULL)
//              gtk_widget_modify_style(drawing_area, style);
//          gtk_rc_style_unref(style);
#endif
#endif
	    fullscreen = 0;
	}
    } else {
	if (value) {
	    if (DEBUG > 1)
		printf("SetFullscreen setting fullscreen\n");
#ifdef GTK_ENABLED
#ifdef GTK2_ENABLED
	    fs_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	    gtk_window_fullscreen(GTK_WINDOW(fs_window));
	    gtk_widget_show(fs_window);
	    gtk_widget_reparent(gtkwidget, fs_window);
	    XReparentWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			    GDK_WINDOW_XWINDOW(gtkwidget->window),
			    GDK_WINDOW_XWINDOW(fs_window->window), 0, 0);

	    gtk_widget_map(gtkwidget);
	    gdisp = gdk_display_get_default();
	    screen = gdk_screen_get_default();
	    gdk_screen_get_monitor_geometry(screen,
					    gdk_screen_get_monitor_at_window
					    (screen, gtkwidget->window),
					    &rect);

	    x = rect.width;
	    y = rect.height;
	    gtk_window_resize(GTK_WINDOW(gtkwidget), rect.width,
			      rect.height);
#endif
#ifdef GTK1_ENABLED
	    fs_window = gtk_window_new(GTK_WINDOW_POPUP);
	    gtk_widget_realize(fs_window);
	    gtk_widget_show(fs_window);
	    XReparentWindow(GDK_WINDOW_XDISPLAY(fs_window->window),
			    GDK_WINDOW_XWINDOW(fs_window->window),
			    GDK_ROOT_WINDOW(), 0, 0);
	    XReparentWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			    GDK_WINDOW_XWINDOW(gtkwidget->window),
			    GDK_WINDOW_XWINDOW(fs_window->window), 0, 0);
	    XMapWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
		       GDK_WINDOW_XWINDOW(gtkwidget->window));
	    x = gdk_screen_width();
	    y = gdk_screen_height();
	    XResizeWindow(GDK_WINDOW_XDISPLAY(fs_window->window),
			  GDK_WINDOW_XWINDOW(fs_window->window), x, y);
	    XResizeWindow(GDK_WINDOW_XDISPLAY(gtkwidget->window),
			  GDK_WINDOW_XWINDOW(gtkwidget->window), x, y);
#endif


	    if (panel_drawn == 1) {
		height = 16;
		width = 21;

		multiplier = 0;

		if (mmsstream == 0) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(rew_event_box),
				   width * multiplier++, y - height);
		}
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(play_event_box),
			       width * multiplier++, y - height);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(pause_event_box),
			       width * multiplier++, y - height);
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(stop_event_box),
			       width * multiplier++, y - height);
		if (mmsstream == 0) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(ff_event_box),
				   width * multiplier++, y - height);
		}
		if (mediaprogress_bar != NULL) {
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(mediaprogress_bar),
				   (width * multiplier + 10),
				   y - height + 2);

		    gtk_widget_set_usize(GTK_WIDGET(mediaprogress_bar),
					 x - (width * (multiplier + 1) +
					      20), height - 4);
		}
		gtk_fixed_move(GTK_FIXED(fixed_container),
			       GTK_WIDGET(fs_event_box),
			       (x - width), y - height);

	    }

	    disp_y = y;
	    disp_x = x;
	    if (movie_height != 0 && movie_width != 0) {
		disp_y = ((long int) (x * movie_height)) / movie_width;
		disp_x = ((long int) (y * movie_width)) / movie_height;
	    } else {
		if (win_width != 0) {
		    disp_y = ((long int) (x * win_height)) / win_width;
		    disp_x = ((long int) (y * win_width)) / win_height;
		}
	    }
	    pos_y = (y - disp_y) / 2;
	    pos_x = (x - disp_x) / 2;
	    if (drawing_area != NULL) {
		if (disp_y <= y) {
		    gtk_widget_set_usize(drawing_area, x, disp_y);
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(drawing_area), 0, pos_y);
		} else {
		    gtk_widget_set_usize(drawing_area, disp_x, y);
		    gtk_fixed_move(GTK_FIXED(fixed_container),
				   GTK_WIDGET(drawing_area), pos_x, 0);
		}
	    }
	    if (DEBUG > 1)
		printf
		    ("x=%i, y=%i, movie_width=%i, movie_height=%i, disp_y=%i, pos_y=%i\n",
		     x, y, movie_width, movie_height, disp_y, pos_y);

	    if (image != NULL)
		gtk_widget_hide(GTK_WIDGET(image));

	    if (progress_bar != NULL)
		gtk_widget_hide(GTK_WIDGET(progress_bar));

	    gtk_widget_set_usize(GTK_WIDGET(status), x - 20, 30);
	    gtk_fixed_move(GTK_FIXED(fixed_container), GTK_WIDGET(status),
			   10, 10);
#ifdef GTK2_ENABLED
	    gtk_widget_modify_bg(gtkwidget, GTK_STATE_NORMAL, &black);
	    if (drawing_area != NULL)
		gtk_widget_modify_bg(drawing_area, GTK_STATE_NORMAL,
				     &black);
#endif
#ifdef GTK1_ENABLED
	    style = gtk_rc_style_new();
	    style->bg[GTK_STATE_NORMAL] = black;
	    style->color_flags[GTK_STATE_NORMAL] =
		(GtkRcFlags) (style->
			      color_flags[GTK_STATE_NORMAL] | GTK_RC_BG);
	    gtk_widget_modify_style(gtkwidget, style);
	    if (drawing_area != NULL)
		gtk_widget_modify_style(drawing_area, style);
	    gtk_rc_style_unref(style);
#endif
#endif
	    fullscreen = 1;
	} else {
	    // do nothing
	    if (DEBUG > 1)
		printf("SetFullscreen doing nothing\n");
	    fullscreen = 0;
	}
    }
#ifdef GTK_ENABLED
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM
				   (menuitem_fullscreen), fullscreen);
#endif

}

void nsPluginInstance::GetShowlogo(PRBool * _retval)
{
    *_retval = (PRBool) showlogo;

}

void nsPluginInstance::SetShowlogo(PRBool value)
{
    showlogo = (int) value;
#ifdef GTK_ENABLED
    if (image != NULL)
	if (showlogo)
	    gtk_widget_show(GTK_WIDGET(image));
	else
	    gtk_widget_hide(GTK_WIDGET(image));

#endif
}

// ==============================
// ! Scriptability related code !
// ==============================
//
// here the plugin is asked by Mozilla to tell if it is scriptable
// we should return a valid interface id and a pointer to
// nsScriptablePeer interface which we should have implemented
// and which should be defined in the corressponding *.xpt file
// in the bin/components folder
NPError nsPluginInstance::GetValue(NPPVariable aVariable, void *aValue)
{
    NPError rv = NPERR_NO_ERROR;

    switch (aVariable) {
    case NPPVpluginScriptableInstance:{
	    // addref happens in getter, so we don't addref here
	    nsIScriptableMplayerPlugin *scriptablePeer =
		getScriptablePeer();
	    if (scriptablePeer) {
		*(nsISupports **) aValue = scriptablePeer;
	    } else
		rv = NPERR_OUT_OF_MEMORY_ERROR;
	}
	break;

    case NPPVpluginScriptableIID:{
	    static nsIID scriptableIID = NS_ISCRIPTABLEMPLAYERPLUGIN_IID;
	    nsIID *ptr = (nsIID *) NPN_MemAlloc(sizeof(nsIID));
	    if (ptr) {
		*ptr = scriptableIID;
		*(nsIID **) aValue = ptr;
	    } else
		rv = NPERR_OUT_OF_MEMORY_ERROR;
	}
	break;

    default:
	break;
    }

    return rv;
}

// ==============================
// ! Scriptability related code !
// ==============================
//
// this method will return the scriptable object (and create it if necessary)
nsScriptablePeer *nsPluginInstance::getScriptablePeer()
{
    if (!mScriptablePeer) {
	mScriptablePeer = new nsScriptablePeer(this);
	if (!mScriptablePeer)
	    return NULL;

	NS_ADDREF(mScriptablePeer);
    }
    // add reference for the caller requesting the object
    NS_ADDREF(mScriptablePeer);
    return mScriptablePeer;
}

nsControlsScriptablePeer *nsPluginInstance::getControlsScriptablePeer()
{
    if (!mControlsScriptablePeer) {
	mControlsScriptablePeer = new nsControlsScriptablePeer(this);
	if (!mControlsScriptablePeer)
	    return NULL;

	NS_ADDREF(mControlsScriptablePeer);
    }
    // add reference for the caller requesting the object
    NS_ADDREF(mControlsScriptablePeer);
    return mControlsScriptablePeer;
}
