/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * Copyright 2008 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: ainfographic.cxx,v $
 * $Revision: 1.12 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_svx.hxx"
#include <svx/sdr/animation/ainfographic.hxx>
#include <svx/sdr/mixer/basicmixer.hxx>
#include <goodies/grfmgr.hxx>
#include <svx/svdograf.hxx>
#include <vcl/svapp.hxx>
#include <svx/sdr/contact/displayinfo.hxx>
#include <svx/sdr/contact/viewobjectcontact.hxx>
#include <svx/sdr/contact/objectcontact.hxx>

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace mixer
	{
		class GraphicMixer : public BasicMixer
		{
			// the associated animationinfo
			sdr::animation::AInfoGraphic&			mrAInfoGraphic;

		protected:
			// setup associated object for paint at given state
			virtual void SetupObject(double fMixerState, sdr::contact::DisplayInfo& rDisplayInfo);

			// restore associated object to original state
			virtual void RestoreObject(sdr::contact::DisplayInfo& rDisplayInfo);

		public:
			// basic constructor.
			GraphicMixer(sdr::animation::AInfoGraphic& rAIGraphic);

			// destructor
			virtual ~GraphicMixer();
		};
	} // end of namespace mixer
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace mixer
	{
		// setup associated object for paint at given state
		void GraphicMixer::SetupObject(double fMixerState, sdr::contact::DisplayInfo& rDisplayInfo)
		{
			const sal_uInt32 nTimeInCycle = (sal_uInt32)(fMixerState * (double)mrAInfoGraphic.GetFullCycleTime());
			sal_uInt32 nStepIndex(mrAInfoGraphic.GetStepIndex(nTimeInCycle));
			//SdrGrafObj& rGrafObj = mrAInfoGraphic.GetSdrGrafObj();

			// get VirtualDevice with prepared output
			BitmapEx aBitmapEx = mrAInfoGraphic.GetBitmapExInState(nStepIndex);

			// set at GrafObj
			rDisplayInfo.GetPaintInfoRec()->mbUseBitmapEx = sal_True;
			rDisplayInfo.GetPaintInfoRec()->maBitmapEx = aBitmapEx;
		}

		// restore associated object to original state
		void GraphicMixer::RestoreObject(sdr::contact::DisplayInfo& rDisplayInfo)
		{
			// reset at GrafObj
			rDisplayInfo.GetPaintInfoRec()->mbUseBitmapEx = sal_False;
			rDisplayInfo.GetPaintInfoRec()->maBitmapEx = BitmapEx();
		}

		GraphicMixer::GraphicMixer(sdr::animation::AInfoGraphic& rAIGraphic)
		:	mrAInfoGraphic(rAIGraphic)
		{
		}

		GraphicMixer::~GraphicMixer()
		{
		}
	} // end of namespace mixer
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////


namespace sdr
{
	namespace animation
	{
		// Access the BasicMixer, plus creation on demand.
		sdr::mixer::BasicMixer* AInfoGraphic::CreateBasicMixer()
		{
			return new sdr::mixer::GraphicMixer((AInfoGraphic&)(*this));
		}

		// Centralized method to get the cycle time for a given animation state.
		// Handles some speclial cases.
		sal_uInt32 AInfoGraphic::GetAnimBitmapWaitTimeInMs(sal_uInt32 nIndex) const
		{
			sal_uInt32 nWaitTime(0L);

			// #128539# secure access to frames
			if(nIndex < mnStepCount)
			{
				const AnimationBitmap& rAnimBitmap = maAnimation.Get(sal_uInt16(nIndex));
				nWaitTime = (rAnimBitmap.nWait * 10);

				// #115934#
				// Take care of special value for MultiPage TIFFs. ATM these shall just
				// show their first page. Later we will offer some switching when object
				// is selected.
				if(ANIMATION_TIMEOUT_ON_CLICK == rAnimBitmap.nWait)
				{
					// ATM the huge value would block the timer, so
					// use a long time to show first page (whole day)
					nWaitTime = 100 * 60 * 60 * 24;
				}
			}

			// Bad trap: There are animated gifs with no set WaitTime (!).
			// In that case use a default value.
			if(0L == nWaitTime)
			{
				nWaitTime = 100L;
			}

			return nWaitTime;
		}

		// calculate the MixerState value for given time
		double AInfoGraphic::GetMixerState(sal_uInt32 nTime, sdr::contact::DisplayInfo& /*rDisplayInfo*/) const
		{
			// #i38495#
			const sal_uInt32 nCurrentLoop((sal_Int32)(nTime / mnFullCycleTime));

			if(mnLoopCount && nCurrentLoop >= mnLoopCount)
			{
				// Display in start state
				return 0.0;
			}

			// calculate time in cycle
			const sal_uInt32 nTimeInCycle(nTime % mnFullCycleTime);

			// divide with mnFullCycleTime
			return ((double)nTimeInCycle / (double)mnFullCycleTime);
		}

		// basic constructor.
		AInfoGraphic::AInfoGraphic(SdrGrafObj& rGrafObj)
		:	mrGrafObj(rGrafObj),
			maVirtualDevice(*Application::GetDefaultDevice()),
			maVirtualDeviceMask(*Application::GetDefaultDevice(), 1L)
		{
			// get the animation
			const GraphicObject& rGraphicObject = GetSdrGrafObj().GetGraphicObject();
			DBG_ASSERT(GRAPHIC_BITMAP == rGraphicObject.GetType() && rGraphicObject.IsAnimated(),
				"AInfoGraphic::AInfoGraphic: graphic is not animated (!)");
			const Graphic& rGraphic = rGraphicObject.GetGraphic();
			maAnimation = rGraphic.GetAnimation();

			// get StepCount
			mnStepCount = (sal_uInt32)maAnimation.Count();

			// get LoopCount
			mnLoopCount = maAnimation.GetLoopCount();

			// calculate full cycle time
			mnFullCycleTime = 0L;

			// #i41470#
			mppGraphicObjects = new GraphicObject*[mnStepCount];

			for(sal_uInt32 a(0L); a < mnStepCount; a++)
			{
				sal_uInt32 nWaitTime(GetAnimBitmapWaitTimeInMs(a));
				mnFullCycleTime += nWaitTime;

				// #i41470#
				mppGraphicObjects[a] = 0L;
			}

			// Prepare VirtualDevice and it's state
			maVirtualDevice.EnableMapMode(sal_False);
			maVirtualDeviceMask.EnableMapMode(sal_False);
			maVirtualDevice.SetOutputSizePixel(maAnimation.GetDisplaySizePixel());
			maVirtualDeviceMask.SetOutputSizePixel(maAnimation.GetDisplaySizePixel());
			maStateOfReplay = 0L;
			AddStateToVirtualDevice(0L);
		}

		// destructor
		AInfoGraphic::~AInfoGraphic()
		{
			for(sal_uInt32 a(0L); a < mnStepCount; a++)
			{
				delete mppGraphicObjects[a];
			}

			delete[] mppGraphicObjects;
		}

		// acces to animation
		::Animation& AInfoGraphic::GetAnimation() const
		{
			return (::Animation&)maAnimation;
		}

		// access to StepCount
		sal_uInt32 AInfoGraphic::GetStepCount() const
		{
			return mnStepCount;
		}

		// access to FullCycleTime
		sal_uInt32 AInfoGraphic::GetFullCycleTime() const
		{
			return mnFullCycleTime;
		}

		// add a animation step to the mixed bitmap
		void AInfoGraphic::AddStateToVirtualDevice(sal_uInt32 nState)
		{
			// #128539# secure access to frames
			if(nState < mnStepCount)
			{
				// prepare next step
				const AnimationBitmap& rAnimBitmap = maAnimation.Get(sal_uInt16(nState));

				switch(rAnimBitmap.eDisposal)
				{
					case DISPOSE_NOT:
					{
						maVirtualDevice.DrawBitmapEx(rAnimBitmap.aPosPix, rAnimBitmap.aBmpEx);
						Bitmap aMask = rAnimBitmap.aBmpEx.GetMask();

						if(aMask.IsEmpty())
						{
							const Point aEmpty;
							const Rectangle aRect(aEmpty, maVirtualDeviceMask.GetOutputSizePixel());
							const Wallpaper aWallpaper(COL_BLACK);
							maVirtualDeviceMask.DrawWallpaper(aRect, aWallpaper);
						}
						else
						{
							BitmapEx aExpandVisibilityMask = BitmapEx(aMask, aMask);
							maVirtualDeviceMask.DrawBitmapEx(rAnimBitmap.aPosPix, aExpandVisibilityMask );
						}

						break;
					}
					case DISPOSE_BACK:
					{
						// #i70772# react on no mask
						const Bitmap aMask(rAnimBitmap.aBmpEx.GetMask());
						const Bitmap aContent(rAnimBitmap.aBmpEx.GetBitmap());

						maVirtualDeviceMask.Erase();
						maVirtualDevice.DrawBitmap(rAnimBitmap.aPosPix, aContent);

						if(aMask.IsEmpty())
						{
							const Rectangle aRect(rAnimBitmap.aPosPix, aContent.GetSizePixel());
							maVirtualDeviceMask.SetFillColor(COL_BLACK);
							maVirtualDeviceMask.SetLineColor();
							maVirtualDeviceMask.DrawRect(aRect);
						}
						else
						{
							maVirtualDeviceMask.DrawBitmap(rAnimBitmap.aPosPix, aMask);
						}

						break;
					}
					case DISPOSE_FULL:
					{
						maVirtualDevice.DrawBitmapEx(rAnimBitmap.aPosPix, rAnimBitmap.aBmpEx);
						break;
					}
					case DISPOSE_PREVIOUS :
					{
						maVirtualDevice.DrawBitmapEx(rAnimBitmap.aPosPix, rAnimBitmap.aBmpEx);
						maVirtualDeviceMask.DrawBitmap(rAnimBitmap.aPosPix, rAnimBitmap.aBmpEx.GetMask());
						break;
					}
				}
			}
		}

		// get a reference to the mixed bitmap in given state
		BitmapEx AInfoGraphic::GetBitmapExInState(sal_uInt32 nState)
		{
			BitmapEx aReturnValue;

			// #128539# secure access to frames
			if(nState < mnStepCount)
			{
				// #i41470#
				GraphicObject* pGraphicObject = mppGraphicObjects[nState];
				
				if(pGraphicObject && GRAPHIC_BITMAP == pGraphicObject->GetGraphic().GetType())
				{
					aReturnValue = pGraphicObject->GetGraphic().GetBitmapEx();
				}
				else
				{
					if(nState == maStateOfReplay)
					{
						// noting to do, return
					}
					else if(nState == maStateOfReplay + 1)
					{
						// prepare next step
						AddStateToVirtualDevice(nState);

						// remember new state
						maStateOfReplay = nState;
					}
					else
					{
						// prepare step nState
						maVirtualDevice.Erase();
						maVirtualDeviceMask.Erase();

						for(sal_uInt32 a(0L); a <= nState; a++)
						{
							AddStateToVirtualDevice(a);
						}

						// remember new state
						maStateOfReplay = nState;
					}

					// create BitmapEx
					Bitmap aMainBitmap = maVirtualDevice.GetBitmap(Point(), maVirtualDevice.GetOutputSizePixel());
					Bitmap aMaskBitmap = maVirtualDeviceMask.GetBitmap(Point(), maVirtualDeviceMask.GetOutputSizePixel());
					aReturnValue = BitmapEx(aMainBitmap, aMaskBitmap);

					// #i41470#
					Graphic aGraphic(aReturnValue);
					mppGraphicObjects[nState] = new GraphicObject(aGraphic);
				}
			}

			// return prepared bitmap
			return aReturnValue;
		}

		// get the step index for given StepTime
		sal_uInt32 AInfoGraphic::GetStepIndex(sal_uInt32 nTimeInCycle) const
		{
			sal_uInt32 nCurrentLoopEndTime(0L);

			for(sal_uInt32 a(0L); a < mnStepCount; a++)
			{
				sal_uInt32 nWaitTime(GetAnimBitmapWaitTimeInMs(a));
				nCurrentLoopEndTime += nWaitTime;

				if(nTimeInCycle < nCurrentLoopEndTime)
				{
					return a;
				}
			}

			return 0L;
		}

		// test if to register at ObjectAnimator again after given time.
		// Fill in the new time value and return accordingly.
		sal_Bool AInfoGraphic::DoRegisterAgain(sal_uInt32 nTime, sal_uInt32& rNewTime, const AnimationState& /*rAssociatedAS*/) const
		{
			// #i38495#
			sal_Bool bRetval(sal_True);
			const sal_uInt32 nCurrentLoop((sal_Int32)(nTime / mnFullCycleTime));

			if(mnLoopCount && nCurrentLoop >= mnLoopCount)
			{
				// End animation
				bRetval = sal_False;
			}
			else
			{
				// calculate time in cycle
				const sal_uInt32 nTimeInCycle(nTime % mnFullCycleTime);

				// find out in which step we are.
				const sal_uInt32 nStepIndex(GetStepIndex(nTimeInCycle));
				sal_uInt32 nWaitTime(GetAnimBitmapWaitTimeInMs(nStepIndex));

				// calculate new time
				rNewTime = nTime + (nWaitTime);
			}

			return bRetval;
		}

		// Decide if animation is allowed for the given VOC.
		sal_Bool AInfoGraphic::IsAnimationAllowed(const sdr::contact::ViewObjectContact& rVOContact) const
		{
			// for accessibility, text animation may be forbidden
			return rVOContact.GetObjectContact().IsGraphicAnimationAllowed();
		}
	} // end of namespace animation
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// eof
