#include <assert.h>

#include <map>

#include "lib/util.h"
#include "watch_maildirs.h"
#include "watcher_maildir.h"
#include "watcher_partial.h"
#include "watcher_base.h"

using std::map;
using std::string;

watcher_base::watcher_base(struct inotify_state* state)
	: inotify_fd(inotify_init()), state(state)
{
	die_if(inotify_fd < 0, "inotify_init");

	// watch for maildir creates and renames and 'maildir' deletion
	// NOTE: This will not detect a maildir move (or a parent move).
	//       Detecting this would take some good work, eg directory symlinks.
	int wd = inotify_add_watch(inotify_fd, state->base_path.c_str(), notify);
	die_if(wd < 0, "inotify_add_watch(\"%s\")", state->base_path.c_str());
}

watcher_base::~watcher_base()
{
	int r;
	if (wd >= 0)
	{
		r = inotify_rm_watch(inotify_fd, wd);
		die_if(r < 0, "inotify_rm_watch(base maildir)\n");
	}
	r = close(inotify_fd);
	die_if(r < 0, "close(inotify_fd)");
}


void watcher_base::process_event(const struct inotify_event& event)
{
	if (event.mask & IN_DELETE_SELF)
		exit(1);

	if (!(event.mask & IN_ISDIR && !strncmp(event.name, ".", 1)))
		return; // event does not involve a Maildir

	if (event.mask & (IN_CREATE | IN_MOVED_TO))
	{
		string dir(event.name);
		string dir_full(mkfilename(state->base_path, dir));
		if (is_maildir(dir_full))
		{
			new watcher_maildir(state, dir);
			send_maildir_modified(maildir_to_imap(dir).c_str());
		}
		else
			new watcher_partial(state, dir);
	}
	else if (event.mask & IN_MOVED_FROM)
	{
		string dir(event.name);
		string imap_name(maildir_to_imap(dir));
		watcher* w = state->get_imap_watcher(imap_name);
		assert(w);
		delete w;
		send_maildir_modified(imap_name.c_str());
	}
}
