/*
 * Copyright (c) Wipro Technologies Ltd, 2003.  All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it would be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 */
/**************************************************************************
 * 
 *    TEST IDENTIFIER	: timer_delete02 
 * 
 *    EXECUTED BY	: anyone
 * 
 *    TEST TITLE	: Basic test for timer_delete(2)
 * 
 *    TEST CASE TOTAL	: 1
 * 
 *    AUTHOR		: Aniruddha Marathe <aniruddha.marathe@wipro.com>
 * 
 *    SIGNALS
 * 	Uses SIGUSR1 to pause before test if option set.
 * 	(See the parse_opts(3) man page).
 *
 *    DESCRIPTION
 *     This is a Phase I test for the timer_delete(2) system call.
 *     It is intended to provide a limited exposure of the system call.
 * 
 * 	Setup:
 *	  Setup signal handling.
 *	  Pause for SIGUSR1 if option specified.
 * 
 * 	Test:
 *	 Loop if the proper options are given.
 *	 Create a POSIX timer
 *	 Execute system call 
 *	 Check return code, if system call failed (return=-1)
 *		Log the errno and Issue a FAIL message.
 *	 Otherwise, Issue a PASS message.
 * 
 * 	Cleanup:
 * 	  Print errno log and/or timing stats if options given
 * 
 * USAGE:  <for command-line>
 * timer_delete02 [-c n] [-e] [-i n] [-I x] [-P x] [-t] [-p]
 * where:
 * 	-c n : Run n copies simultaneously. 
 *	-e   : Turn on errno logging.
 *	-i n : Execute test n times.
 *	-I x : Execute test for x seconds.
 *	-p   : Pause for SIGUSR1 before starting
 *	-P x : Pause for x seconds between iterations.
 *	-t   : Turn on syscall timing.
 *
 *RESTRICTIONS:
 * None
 *****************************************************************************/

#include "test.h"
#include "usctest.h"
#include <errno.h>
#include <syscall.h>
#include <time.h>
#include <signal.h>

#ifndef _syscall2
#include <linux/unistd.h>
#endif


#ifndef __NR_timer_create
#if defined(__i386__)
#define __NR_timer_create 259
#elif defined(__ppc__)
#define __NR_timer_create 240
#elif defined(__powerpc64__)
#define __NR_timer_create 240
#elif defined(__x86_64__)
#define __NR_timer_create 222
#endif
#endif

#ifndef __NR_timer_delete
#if defined(__i386__)
#define __NR_timer_delete (__NR_timer_create + 4)
#elif defined(__ppc__)
#define __NR_timer_delete 244
#elif defined(__powerpc64__)
#define __NR_timer_delete 244
#elif defined(__x86_64__)
#define __NR_timer_delete 226
#endif
#endif

/* We are declaring timer_create and timer_delete as weak symbols thorugh
 * following definitions. In newer glibc versions, these functions
 * should be present. At that time, glibc definition will supersede the 
 * definition in this code which is done using _syscall macro
 */ 
#pragma weak timer_create
#pragma weak timer_delete

static void setup();
static void cleanup();

char *TCID = "timer_delete02";	/* Test program identifier.    */
int TST_TOTAL = 1;		/* Total number of test cases. */
extern int Tst_count;		/* Test Case counter for tst_* routines */

/* syscall macros are used to register timer_create and timer_delete functions
 * as system calls
 */ 
_syscall3(int, timer_create, clockid_t, which_clock, struct sigevent *,
	timer_event_spec, timer_t *, timer_id);
_syscall1(int, timer_delete, timer_t, timer_id);

int
main(int ac, char **av)
{
	int lc;		/* loop counter */
	char *msg;	/* message returned from parse_opts */
	timer_t timer_id;

	/* parse standard options */
	if ((msg = parse_opts (ac, av, (option_t *) NULL, NULL)) != 
			(char *) NULL) {
		tst_brkm(TBROK, tst_exit, "OPTION PARSING ERROR - %s", msg);
	}

	/* perform global setup for test */
	setup();

	/* check looping state if -i option given */
	for (lc = 0; TEST_LOOPING(lc); lc++) {

		/* reset Tst_count in case we are looping. */
		Tst_count = 0;

		/* Create a Posix timer */
		if (timer_create(CLOCK_REALTIME, NULL, &timer_id) < 0) {

			/* If timer_create system call is not implemented
			 * in the running kernel, this will fail with ENOSYS
			 */ 
			Tst_count = TST_TOTAL;
			perror("timer_create");
			tst_brkm(TBROK, cleanup, "timer_delete can't be"
					" tested because timer_create failed");
		}

		TEST(timer_delete(timer_id));

		if (TEST_RETURN == -1) {
			/* If timer_delete system call is not implemented
			 * in the running kernel, test will fail with ENOSYS
			 */ 
			if (TEST_ERRNO == ENOSYS) {
				Tst_count = TST_TOTAL;
				perror("timer_delete");
				tst_brkm(TBROK, cleanup, "");
			}	
			TEST_ERROR_LOG(TEST_ERRNO);
			tst_resm(TFAIL, "timer_delete(2) Failed and set errno" 
					" to %d", TEST_ERRNO);
		} else {
			tst_resm(TPASS, "timer_delete(2) Passed");
		}
	}	/* End for TEST_LOOPING */

	/* Clean up and exit */
	cleanup();

	/* NOTREACHED */
	return 0;
}

/* setup() - performs all ONE TIME setup for this test */
void
setup()
{
	/* capture signals */
	tst_sig(NOFORK, DEF_HANDLER, cleanup);

	/* Pause if that option was specified */
	TEST_PAUSE;
}	/* End setup() */

/*
 * cleanup() - Performs one time cleanup for this test at
 * completion or premature exit
 */

void
cleanup()
{
	/*
	* print timing stats if that option was specified.
	* print errno log if that option was specified.
	*/
	TEST_CLEANUP;

	/* exit with return code appropriate for results */
	tst_exit();
}	/* End cleanup() */
