#ifndef KADU_SOUND_H
#define KADU_SOUND_H

#include <qobject.h>
#include <qstring.h>
#include <qdatetime.h>
#include <qstringlist.h>
#include <qmap.h>
#include <qthread.h>
#include <qmutex.h>
#include <qsemaphore.h>

#include "config_file.h"
#include "gadu.h"
#include "misc.h"
#include "message_box.h"
#include "modules.h"
#include "themes.h"
#include "usergroup.h"
#include "userlist.h"
#include "../notify/notify.h"
/**
 * @defgroup sound Sound
 * The sound module.
 * @{
 */

/**
	Uoglniony deskryptor urzdzenia dwikowego.
**/
typedef void* SoundDevice;

/**
**/
enum SoundDeviceType {RECORD_ONLY, PLAY_ONLY, PLAY_AND_RECORD};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SamplePlayThread : public QObject, public QThread
{
	Q_OBJECT

	private:
		SoundDevice Device;
		const int16_t* Sample;
		int SampleLen;
		bool Stopped;
		QSemaphore PlayingSemaphore;
		QSemaphore SampleSemaphore;

	protected:
		virtual void run();
		virtual void customEvent(QCustomEvent* event);

	public:
		SamplePlayThread(SoundDevice device);
		void playSample(const int16_t* data, int length);
		void stop();

	signals:
		void samplePlayed(SoundDevice device);
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SampleRecordThread : public QObject, public QThread
{
	Q_OBJECT

	private:
		SoundDevice Device;
		int16_t* Sample;
		int SampleLen;
		bool Stopped;
		QSemaphore RecordingSemaphore;
		QSemaphore SampleSemaphore;

	protected:
		virtual void run();
		virtual void customEvent(QCustomEvent* event);

	public:
		SampleRecordThread(SoundDevice device);
		void recordSample(int16_t* data, int length);
		void stop();

	signals:
		void sampleRecorded(SoundDevice device);
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SndParams
{
	public:
		SndParams(QString fm = QString::null, bool volCntrl = false, float vol = 1);
		SndParams(const SndParams &p);

		QString filename;
		bool volumeControl;
		float volume;
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SoundPlayThread : public QThread
{
	public:
		SoundPlayThread();
		~SoundPlayThread();
		void run();
		void tryPlay(const char *path, bool volCntrl=false, float volume=1.0);
		void endThread();

	private:
		static bool play(const char *path, bool volCntrl=false, float volume=1.0);
		QMutex mutex;
		QSemaphore *semaphore;
		bool end;
		QValueList<SndParams> list;
};

class SoundManager : public Notifier
{
    Q_OBJECT
	private:
		Themes *themes;
		friend class SamplePlayThread;
		friend class SampleRecordThread;
		QTime lastsoundtime;
		bool mute;
		QMap<SoundDevice, SamplePlayThread*> PlayingThreads;
		QMap<SoundDevice, SampleRecordThread*> RecordingThreads;
		SoundPlayThread *play_thread;

		int simple_player_count;
		virtual void connectNotify(const char *signal);
		virtual void disconnectNotify(const char *signal);

	private slots:
		void newChat(Protocol *protocol, UserListElements senders, const QString &msg, time_t t);
		void newMessage(Protocol *protocol, UserListElements senders, const QString &msg, time_t t, bool &grab);
		void connectionError(Protocol *protocol, const QString &message);
		void userChangedStatusToAvailable(const QString &protocolName, UserListElement);
		void userChangedStatusToBusy(const QString &protocolName, UserListElement);
		void userChangedStatusToInvisible(const QString &protocolName, UserListElement);
		void userChangedStatusToNotAvailable(const QString &protocolName, UserListElement);

		/* from i ule s ignorowane, message wskazuje na plik z dwikiem do odtworzenia
		 * jeeli message==QString::null, to odtwarzany jest standardowy dwik dla tego typu
		 * jeeli mapa jest!=NULL brane s z niej nastpujce wartoci:
		 *		"Force"           - bool (wymuszenie odtwarzania mimo wyciszenia)
		 */
		void message(const QString &from, const QString &message, const QMap<QString, QVariant> *parameters, const UserListElement *ule);
		virtual void externalEvent(const QString &notifyType, const QString &msg, const UserListElements &ules);

	public slots:
		void play(const QString &path, bool force=false);
		void play(const QString &path, bool volCntrl, double vol);
		void setMute(const bool& enable);

	public:

		SoundManager(const QString& name, const QString& configname);
		~SoundManager();

		Themes *theme();

		bool isMuted() const;
		int timeAfterLastSound() const;
		/**
			Otwiera urzdzenie dwikowe do operacji
			odtwarzania i nagrywania sampli.
			Niektre implementacje pozwalaj na otwarcie
			wielu niezalenie dziaajcych "pocze"
			z urzdzeniami. Wystarczy wtedy kilkukrotnie
			wywoa t metod.
			Emituje sygna openDeviceImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			@param type
			@param sample_rate sample rate - np. 8000 lub 48000
			@param channels ilo kanaw: 1 - mono, 2 - stereo
			@return uoglniony deskryptor urzdzenia lub NULL jeli otwarcie si nie powiodo.
		**/
		SoundDevice openDevice(SoundDeviceType type, int sample_rate, int channels = 1);
		/**
			Zamyka urzdzenie dwikowe otwarte za pomoc
			metody openDevice().
			Niektre implementacje pozwalaj na otwarcie
			wielu niezalenie dziaajcych "pocze"
			z urzdzeniami. Kade otwarte poczenie naley
			zamkn za pomoc tej metody.
			Jeli wczylimy operacje nieblokujce to
			metoda ta czeka na zakoczenie trwajcych operacji
			i koczy dziaanie wtkw.
			Emituje sygna closeDeviceImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			@param device uoglniony deskryptor urzdzenia.
		**/
		void closeDevice(SoundDevice device);
		/**
			Powouje do ycia wtki zajmujce si odtwarzaniem
			i nagrywaniem prbek dla danego poczenia z
			urzdzeniem dwikowym.
			Od tej chwili playSample() i recordSample()
			bd operacjami nieblokujcymi.
			@param device uoglniony deskryptor urzdzenia.
		**/
		void enableThreading(SoundDevice device);
		/**
			Standardowo po przekazaniu odtwarzanego sampla
			do sterownika dwikowego program (w trybie blokujcym)
			lub wtek odtwarzajcy (w trybie nieblokujcym) czeka
			na zakoczenie emitowania dwiku przez sterownik, aby
			zagwarantowa, e wyjcie z funkcji playSample() (w trybie
			blokujcym) lub wyemitowanie sygnau samplePlayed() (w
			trybie nieblokujcym) nastpi dopiero po fizycznym
			odegraniu dwiku. Inaczej mogo by to nastpi dopiero
			przy wywoaniu metody closeDevice() co ma rne nieprzyjemne
			efekty uboczne.
			Czasem przy odtwarzaniu kilku maych sampli jeden po drugim
			powoduje to powstanie przerw midzy nimi. Aby tego unikn
			moemy uy setFlushingEnabled(device, false) umoliwiajc
			w ten sposb pynne odtwarzanie kilku sampli bezporednio
			po sobie.
		**/
		void setFlushingEnabled(SoundDevice device, bool enabled);
		/**
			Odtwarza prbk dwikow. Standardowo jest to
			operacja blokujca. Moe by wywoana z innego
			wtku (a nawet powinna).
			Emituje sygna playSampleImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			Po uprzednim wywoaniu enableThreading() dziaanie
			metoda jest nieblokujca i przekazuje jedynie polecenie
			odtwarzania do wtku.
			W takim wypadku naley uwaa, aby nie zwolni pamici
			zajmowanej przez dane sampla zanim odtwarzanie si nie
			zakoczy.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik do danych sampla
			@param length dugo danych sampla (w bajtach)
			@return true jeli odtwarzanie zakoczyo si powodzeniem.
		**/
		bool playSample(SoundDevice device, const int16_t* data, int length);
		/**
			Nagrywa prbk dwikow. Standardowo jest to
			operacja blokujca. Moe by wywoana z innego
			wtku (a nawet powinna).
			Emituje sygna recordSampleImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			Po uprzednim wywoaniu enableThreading() dziaanie
			metoda jest nieblokujca i przekazuje jedynie polecenie
			nagrywania do wtku.
			W takim wypadku naley uwaa, aby nie zwolni pamici
			bufora na dane sampla zanim nagrywanie si nie
			zakoczy.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik na bufor dla danych sampla
			@param length dugo sampla do nagrania (wielko bufora w bajtach)
			@return true jeli nagrywanie zakoczyo si powodzeniem.
		**/
		bool recordSample(SoundDevice device, int16_t* data, int length);

	signals:
		void playSound(const QString &sound, bool volCntrl, double vol);
		/**
			Sygna emitowany gdy odtwarzanie sampla si
			zakoczyo (odnosi si tylko do sytuacji gdy
			wczone s operacje nieblokujce).
		**/
		void samplePlayed(SoundDevice device);
		/**
			Sygna emitowany gdy nagrywanie sampla si
			zakoczyo (odnosi si tylko do sytuacji gdy
			wczone s operacje nieblokujce).
		**/
		void sampleRecorded(SoundDevice device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			otwarcia urzdzenia dwikowego do operacji
			odtwarzania i nagrywania sampli.
			@param type
			@param sample_rate sample rate - np. 8000 lub 48000
			@param channels ilo kanaw: 1 - mono, 2 - stereo
			@device zwrcony uoglniony deskryptor urzdzenia lub NULL jeli otwarcie si nie powiodo.
		**/
		void openDeviceImpl(SoundDeviceType type, int sample_rate, int channels, SoundDevice& device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			Zamknicia urzdzenia dwikowego otwartegp za pomoc
			metody openDevice().
			@param device uoglniony deskryptor urzdzenia.
		**/
		void closeDeviceImpl(SoundDevice device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			odtworzenia prbki dwikowej.
			Modu powinien odda sterowanie dopiero po
			odtworzeniu prbki.
			Sygna zazwyczaj bdzie emitowany z innego
			wtku i slot musi by do tego przystosowany.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik do danych sampla
			@param length dugo danych sampla (w bajtach)
			@param result zwrcony rezultat operacji - true jeli odtwarzanie zakoczyo si powodzeniem.
		**/
		void playSampleImpl(SoundDevice device, const int16_t* data, int length, bool& result);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			nagrania prbki dwikowej.
			Modu powinien odda sterowanie dopiero po
			nagraniu prbki.
			Sygna zazwyczaj bdzie emitowany z innego
			wtku i slot musi by do tego przystosowany.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik na bufor dla danych sampla
			@param length dugo sampla do nagrania (wielko bufora w bajtach)
			@param result zwrcony rezultat operacji - true jeli nagrywanie zakoczyo si powodzeniem.
		**/
		void recordSampleImpl(SoundDevice device, int16_t* data, int length, bool& result);
		/**
		**/
		void setFlushingEnabledImpl(SoundDevice device, bool enabled);
};

extern SoundManager* sound_manager;
/** @} */

#endif
