/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlistbox.h>
#include <qmenubar.h>
#include <qpopupmenu.h>

#include "powerkadu.h"
#include "word_fix.h"
#include "translator.h"
#include "anonymous_check.h"
#include "infos.h"
#include "cmdline.h"
#include "ekg_cmds.h"
#include "antistring.h"
#include "i_hide_from_u.h"
#include "autostatus.h"
#include "split.h"
#include "mime_tex.h"
#include "auto_hide.h"
#include "cenzor.h"
#include "about_dialog.h"
#include "parser_extender.h"

#include "kadu.h"
#include "config_file.h"
#include "config_dialog.h"
#include "misc.h"
#include "modules.h"
#include "chat_manager.h"
#include "debug.h"

PowerKadu* powerKadu;

extern "C" int powerkadu_init()
{
	kdebugf();
	powerKadu = new PowerKadu();
	powerKadu->init();
	kdebugf2();
	return 0;
}

extern "C" void powerkadu_close()
{
	kdebugf();
	delete powerKadu;
	kdebugf2();
}

//-----------------------------------------------------------------------------------

PowerKadu::PowerKadu()
 : QObject()
{
	kdebugf();

	// Config Dialog
	config = new ConfigFile("powerkadu.conf");

	ConfigDialog::registerSlotOnCreateTab("PowerKadu", this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnCloseTab("PowerKadu", this, SLOT(onDestroyConfig()));
	ConfigDialog::registerSlotOnApplyTab("PowerKadu", this, SLOT(onApplyConfig()));
	
	ConfigDialog::addTab("PowerKadu", dataPath("kadu/modules/data/powerkadu/powerkadu_32x32.png"));
	ConfigDialog::addVGroupBox("PowerKadu", "PowerKadu", 
			QT_TRANSLATE_NOOP("@default", "PowerKadu messages"));
	ConfigDialog::addColorButton("PowerKadu", "PowerKadu messages", 
			QT_TRANSLATE_NOOP("@default", "Background color"), "powerkadu_msg_bg_color",
			QColor(255, 255, 255));
	ConfigDialog::addColorButton("PowerKadu", "PowerKadu messages",
			QT_TRANSLATE_NOOP("@default", "Font color"), "powerkadu_msg_fg_color",
			QColor(0, 0, 0));

	kdebugf2();
}


PowerKadu::~PowerKadu()
{
	kdebugf();

	// Internal objects
	delete wordFix;
	delete morse;
	delete basic1337;
	delete alt1337;
	delete anonymousCheck;
	delete infos;
	delete ekgCmds;
	delete cmdLine;
	delete antistring;
	delete iHideFromU;
	delete autostatus;
	delete split;
	delete mimeTeX;
	delete autoHide;
	delete cenzor;
	delete parserExtender;
	
	delete menu;
	
	// Some PK classes might use it, so config is the last deleted object
	delete config;

	// Config dialog
	ConfigDialog::removeControl("PowerKadu", "Background color");
	ConfigDialog::removeControl("PowerKadu", "Font color");
	ConfigDialog::removeControl("PowerKadu", "PowerKadu messages");
	ConfigDialog::removeTab("PowerKadu");

	ConfigDialog::unregisterSlotOnApplyTab("PowerKadu", this, SLOT(onApplyConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("PowerKadu", this, SLOT(onDestroyConfig()));
	ConfigDialog::unregisterSlotOnCreateTab("PowerKadu", this, SLOT(onCreateConfig()));

	kdebugf2();
}

void PowerKadu::init()
{
	kdebugf();

	// Main menu
	createMainMenu();

	// Internal objects
	cmdLine = new CmdLine();
	ekgCmds = new EKGCmds(this, "ekg commands");
	wordFix = new WordFix(this);
	morse = new Translator("morse", this, "morse translator");
	basic1337 = new Translator("1337", this, "basic1337 translator");
	alt1337 = new Translator("alt1337", this, "alt1337 translator");
	anonymousCheck = new AnonymousCheck(this, "check anonymous");
	infos = new Infos(this, "infos");
	antistring = new Antistring(this);
	iHideFromU = new IHideFromU(this, "i_hide_from_u");
	autostatus = new Autostatus();
	split = new Split(this, "split");
	mimeTeX = new MimeTeX(this, "mime_tex");
	autoHide = new AutoHide(this, "auto_hide");
	cenzor = new Cenzor(this);
	parserExtender = new ParserExtender();

	// After all modules has been loaded:
	cmdLine->sortCmds();

	// "About PowerKadu..." dialog
	menu->insertSeparator();
	menu->insertItem(tr("&About PowerKadu..."), this, SLOT(onAboutPowerKadu()));

	kdebugf2();
}

void PowerKadu::onCreateConfig()
{
	kdebugf();
	modules_manager->moduleIncUsageCount("powerkadu");

	wordFix->onCreateConfig();
	antistring->onCreateConfig();

	kdebugf2();
}

void PowerKadu::onApplyConfig()
{
	kdebugf();
	config->sync();
	kdebugf2();
}

void PowerKadu::onDestroyConfig()
{
	kdebugf();
	onApplyConfig();
	modules_manager->moduleDecUsageCount("powerkadu");

	wordFix->onDestroyConfig();
	antistring->onDestroyConfig();

	kdebugf2();
}

void PowerKadu::onAboutPowerKadu()
{
	kdebugf();
	(new AboutDialog("about_dialog", TRUE))->exec();
	kdebugf2();
}

ConfigFile* PowerKadu::cfg()
{
	kdebugf();
	return config;
	kdebugf2();
}

CmdLine* PowerKadu::cmds()
{
	kdebugf();
	return cmdLine;
	kdebugf2();
}

QString PowerKadu::HTMLColorStr(const QColor &color)
{
	kdebugf();
	QString result;
	if(color.red() < 0x0F)
		result = "\"#0" + QString::number(color.red(), 16).upper();
	else
		result = "\"#" + QString::number(color.red(), 16).upper();
	if(color.green() < 0x0F)
		result += "0" + QString::number(color.green(), 16).upper();
	else
		result +=  QString::number(color.green(), 16).upper();
	if(color.blue() < 0x0F)
		result += "0" + QString::number(color.blue(), 16).upper() + "\"";
	else
		result += QString::number(color.blue(), 16).upper() + "\"";
	return result;
}

void PowerKadu::createMainMenu()
{
	kdebugf();
	QMenuBar *menuBar = kadu->menuBar();
	menu = new QPopupMenu();
	menuBar->insertItem("&PowerKadu", menu);
	kdebugf2();
}

QPopupMenu *PowerKadu::mainMenu()
{
	kdebugf();
	return menu;
}

void PowerKadu::showPkMsg(Chat* chat, QString msg)
{
	kdebugf();
	QColor tmpColor;
	QColor bgColor = config_file.readColorEntry("PowerKadu", "powerkadu_msg_bg_color", &tmpColor);
	QColor myChatBgColor = config_file.readColorEntry("Look", "ChatMyBgColor");
	QColor color = config_file.readColorEntry("PowerKadu", "powerkadu_msg_fg_color", &tmpColor);
	
	QString msgText = "<table bgcolor=" + HTMLColorStr(bgColor) + " border=1 width=100%><tr><td><font color=" +
		HTMLColorStr(color) + ">" + PK() + msg + "</font></td></tr></table>";
		
	ChatColors chatColors(myChatBgColor, tmpColor, bgColor);
	ChatMessage *tmpMsg = new ChatMessage(msgText, chatColors);
	kdebugm(KDEBUG_INFO, "msg: %s\n", msgText.data());
	QValueList<ChatMessage*> messages;
	messages.append(tmpMsg);
	chat->scrollMessages(messages);
	kdebugf2();
}

void PowerKadu::showPkMsg(UserGroup *users, QString msg)
{
	kdebugf();
	Chat *chat = chat_manager->findChat(users);
	showPkMsg(chat, msg);
	kdebugf2();
}

QString PowerKadu::PK()
{
	kdebugf();
	return "<b>[PK]</b> ";
	kdebugf2();
}
