/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <qlabel.h>
#include <qspinbox.h>

#include "advanced_userlist.h"
#include "config_dialog.h"
#include "config_file.h"
#include "debug.h"
#include "kadu.h"
#include "kadu_parser.h"
#include "misc.h"
#include "pending_msgs.h"
#include "userbox.h"
#include "userinfo.h"
#include "userlist.h"

static int compareByPriority(const UserListElement &u1, const UserListElement &u2)
{
	return u2.data("Priority").toInt() - u1.data("Priority").toInt();
}

static int compareByPending(const UserListElement &u1, const UserListElement &u2)
{
	return int(pending.pendingMsgs(u2)) - int(pending.pendingMsgs(u1));
}

static QString getPriority(const UserListElement &elem)
{
	return elem.data("Priority").toString();
}

AdvancedUserList::AdvancedUserList()
{
	config_file.addVariable("AdvUserList", "Order", "Pending,Status,Priority,AltNick");
	order = QStringList::split(',', config_file.readEntry("AdvUserList", "Order"));

	userlist->addPerContactNonProtocolConfigEntry("priority", "Priority");

	// zliczamy uytkownikw bez priorytetu
	int cnt = 0;
	CONST_FOREACH(user, *userlist)
		if ((*user).data("Priority").isNull())
			++cnt;

	// i ustawiamy im priorytet
	int i = 1;
	FOREACH(user, *userlist)
		if ((*user).data("Priority").isNull())
			(*user).setData("Priority", int(0), true, i++ == cnt);

	connect(userlist, SIGNAL(userAdded(UserListElement, bool, bool)),
			this,       SLOT(userAdded(UserListElement, bool, bool)));

	const QValueList<UserBox *> &userboxes = UserBox::userboxes();
	CONST_FOREACH(userbox, userboxes)
		userboxCreated(*userbox);
	connect(&UserBox::createNotifier, SIGNAL(objectCreated(QObject *)), this, SLOT(userboxCreated(QObject *)));
	connect(&UserInfo::createNotifier, SIGNAL(objectCreated(QObject *)), this, SLOT(userInfoWindowCreated(QObject *)));

	ConfigDialog::addTab(QT_TRANSLATE_NOOP("@default", "User List"), "UserListTab", Advanced);
	ConfigDialog::addVBox("User List", "User List", "vbox");
		ConfigDialog::addVBox("User List", "vbox", "vbox2");
			ConfigDialog::addLabel("User List",   "vbox2", QT_TRANSLATE_NOOP("@default", "Sorting function priority"));
			ConfigDialog::addListBox("User List", "vbox2", "sorting functions");
		ConfigDialog::addHBox("User List", "vbox", "vbox1");
			ConfigDialog::addPushButton("User List", "vbox1", QT_TRANSLATE_NOOP("@default", "Up"));
			ConfigDialog::addPushButton("User List", "vbox1", QT_TRANSLATE_NOOP("@default", "Down"));

	ConfigDialog::connectSlot("User List", "Up",   SIGNAL(clicked()), this, SLOT(upButtonClicked()));
	ConfigDialog::connectSlot("User List", "Down", SIGNAL(clicked()), this, SLOT(downButtonClicked()));

	ConfigDialog::registerSlotOnCreateTab("User List", this, SLOT(onCreateTabAdvUserList()));
	ConfigDialog::registerSlotOnApplyTab("User List", this, SLOT(onApplyTabAdvUserList()));

	KaduParser::registerTag("priority", getPriority);
}

AdvancedUserList::~AdvancedUserList()
{
	KaduParser::unregisterTag("priority", getPriority);

	ConfigDialog::unregisterSlotOnApplyTab("User List", this, SLOT(onApplyTabAdvUserList()));
	ConfigDialog::unregisterSlotOnCreateTab("User List", this, SLOT(onCreateTabAdvUserList()));

	ConfigDialog::disconnectSlot("User List", "Down", SIGNAL(clicked()), this, SLOT(downButtonClicked()));
	ConfigDialog::disconnectSlot("User List", "Up",   SIGNAL(clicked()), this, SLOT(upButtonClicked()));

			ConfigDialog::removeControl("User List", "Down");
			ConfigDialog::removeControl("User List", "Up");
		ConfigDialog::removeControl("User List", "vbox1");
			ConfigDialog::removeControl("User List", "sorting functions");
			ConfigDialog::removeControl("User List", "Sorting function priority");
		ConfigDialog::removeControl("User List", "vbox2");
	ConfigDialog::removeControl("User List", "vbox");
	ConfigDialog::removeTab("User List");

	disconnect(&UserInfo::createNotifier, SIGNAL(objectCreated(QObject *)), this, SLOT(userInfoWindowCreated(QObject *)));
	disconnect(&UserBox::createNotifier, SIGNAL(objectCreated(QObject *)), this, SLOT(userboxCreated(QObject *)));

	disconnect(userlist, SIGNAL(userAdded(UserListElement, bool, bool)),
				this,      SLOT(userAdded(UserListElement, bool, bool)));
	const QValueList<UserBox *> &userboxes = UserBox::userboxes();
	CONST_FOREACH(userbox, userboxes)
	{
		(*userbox)->removeCompareFunction("Priority");
		(*userbox)->removeCompareFunction("Pending");
	}
}

void AdvancedUserList::userAdded(UserListElement elem, bool massively, bool last)
{
	elem.setData("Priority", int(0), massively, last);
}

void AdvancedUserList::userboxCreated(QObject *new_object)
{
	UserBox *box = static_cast<UserBox *>(new_object);
	box->addCompareFunction("Pending", tr("Compares by pending messages (treats contacts with pending messages as more important)"), compareByPending);
	box->addCompareFunction("Priority", tr("Compares priorities"), compareByPriority);

	int idx = 0;
	// lipne sortowanie bbelkowe, pniej si to napisze porzdnie
	CONST_FOREACH(funId, order)
	{
		while (box->compareFunctions()[idx].id != *funId)
			if (!box->moveUpCompareFunction(*funId))
			{
				--idx;
				break;
			}
		++idx;
	}
}

void AdvancedUserList::userInfoWindowCreated(QObject *new_object)
{
	kdebugf();
	UserInfo *info = static_cast<UserInfo *>(new_object);
	connect(info, SIGNAL(updateClicked(UserInfo *)), this, SLOT(updateClicked(UserInfo *)));
	QWidget *space = static_cast<QWidget*>(info->child("space_for_advanced_userlist"));
	if (!space)
		space = info;
	new QLabel(tr("Priority"), space);
	(new QSpinBox(-10, 10, 1, space, "priority_spinbox"))->setValue(info->user().data("Priority").toInt());
	kdebugf2();
}

void AdvancedUserList::updateClicked(UserInfo *info)
{
	kdebugf();
	int val = static_cast<QSpinBox *>(info->child("priority_spinbox"))->value();
	if (info->user().data("Priority").toInt() != val)
	{
		info->user().setData("Priority", val);
		UserBox::refreshAllLater();
	}
	kdebugf2();
}

void AdvancedUserList::onCreateTabAdvUserList()
{
	kdebugf();
	newOrder = order;
	refreshFunsInConfig();
	kdebugf2();
}

void AdvancedUserList::onApplyTabAdvUserList()
{
	kdebugf();
	order = newOrder;
	config_file.writeEntry("AdvUserList", "Order", order.join(","));
	const QValueList<UserBox *> &userboxes = UserBox::userboxes();
	CONST_FOREACH(userbox, userboxes)
		userboxCreated(*userbox);
	kdebugf2();
}

void AdvancedUserList::upButtonClicked()
{
	kdebugf();
	QListBox *lb = ConfigDialog::getListBox("User List", "sorting functions");
	QValueList <UserBox::CmpFuncDesc> cmpFuns = kadu->userbox()->compareFunctions ();
	QString desc = lb->currentText();
	CONST_FOREACH(fun, cmpFuns)
		if ((*fun).description == desc)
		{
			unsigned int idx = 0;
			CONST_FOREACH(id, newOrder)
			{
				if ((*fun).id == (*id))
				{
					if (idx > 0)
					{
						QString id1 = newOrder[idx - 1];
						newOrder[idx - 1] = newOrder[idx];
						newOrder[idx] = id1;
					}
					break;
				}
				++idx;
			}
			break;
		}
	refreshFunsInConfig();
	kdebugf2();
}

void AdvancedUserList::downButtonClicked()
{
	kdebugf();
	QListBox *lb = ConfigDialog::getListBox("User List", "sorting functions");
	QValueList <UserBox::CmpFuncDesc> cmpFuns = kadu->userbox()->compareFunctions ();
	QString desc = lb->currentText();
	CONST_FOREACH(fun, cmpFuns)
		if ((*fun).description == desc)
		{
			unsigned int idx = 0;
			CONST_FOREACH(id, newOrder)
			{
				if ((*fun).id == (*id))
				{
					if (idx + 1 < newOrder.size())
					{
						QString id1 = newOrder[idx + 1];
						newOrder[idx + 1] = newOrder[idx];
						newOrder[idx] = id1;
					}
					break;
				}
				++idx;
			}
			break;
		}
	refreshFunsInConfig();
	kdebugf2();
}

void AdvancedUserList::refreshFunsInConfig()
{
	kdebugf();
	QListBox *lb = ConfigDialog::getListBox("User List", "sorting functions");
	QValueList <UserBox::CmpFuncDesc> cmpFuns = kadu->userbox()->compareFunctions ();
	QString selected = lb->currentText();
	//kdebugm(KDEBUG_INFO, "sel: %s\n", selected.local8Bit().data());
	lb->clear();
	CONST_FOREACH(id, newOrder)
		CONST_FOREACH(fun, cmpFuns)
			if ((*id) == (*fun).id)
			{
				lb->insertItem((*fun).description);
				break;
			}
	QListBoxItem *selItem;
	if (selected.isEmpty())
		selItem = lb->item(0);
	else
		selItem = lb->findItem(selected, Qt::ExactMatch);
	if (selItem)
		lb->setSelected(selItem, true);
	kdebugf2();
}
