/***************************************************************************

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

***************************************************************************
Copyright (C) 2005 by 
Pan Wojtas (Wojtek Sulewski)
wojteksulewski <at> op.pl
gg: 2087202

***************************************************************************/


#include <qcheckbox.h>
#include <qspinbox.h>

#include "osd_manager.h"

#include "config_dialog.h"
#include "chat_manager.h"
#include "kadu_parser.h"
#include "misc.h"
#include "kadu.h"
#include "userbox.h" //mapToGlobal
#include "debug.h"

OSDManager::OSDManager(QWidget *parent, const char *name) : Notifier(parent, name)
{
	kdebugf();
	
	height = 0;
	tipWidget = NULL;
	
	reload();


	connect(chat_manager, SIGNAL(chatCreated(const UserGroup *)), this, SLOT(chatCreated(const UserGroup *)));
	
	kdebugf2();
}

OSDManager::~OSDManager()
{
	disconnect(chat_manager, SIGNAL(chatCreated(const UserGroup *)), this, SLOT(chatCreated(const UserGroup *)));

	deleteAll();
}

void OSDManager::chatCreated(const UserGroup *group)
{
	kdebugf();

	FOREACH(i, *group)
	{
		FOREACH(j, widgets)
		{
			if ((*j)->getUsers().contains(*i))
			{
				timeout((*j)->getID(), true);
				break;
			}
		}
	}

	kdebugf2();
}

void OSDManager::reload(void)
{
	kdebugf();
	
	
	deleteAll();
	new_osd = config_file.readUnsignedNumEntry("osdhints","NewOSD", 0);
	corner = config_file.readUnsignedNumEntry("osdhints", "Corner");
	separator = config_file.readUnsignedNumEntry("osdhints","OSD_separator");
	if (config_file.readBoolEntry("osdhints", "Use translucency", false))
		translucency_level = config_file.readNumEntry("osdhints", "Translucency level", 80) * 0.01;
	else
		translucency_level = 1;
	def_x = config_file.readNumEntry("osdhints", "OSD_x", 300);
	def_y = config_file.readNumEntry("osdhints", "OSD_y", 300);
	
	kdebugf2();
}

void OSDManager::addOSD(QString text, const QFont font, const QColor fg_color, const QColor bg_color, const QColor border_color, const unsigned int timeout, const int maskEffect, const UserListElements senders)
{
	kdebugf();

	if (new_osd == 4)
		deleteAll();

	widgets.append(new OSDWidget(0));
	unsigned int id = widgets.count() - 1;
	widgets.at(id)->setFont(font);
	
	OSDWidget::OSDData data;
	
	HtmlDocument doc;
	doc.parseHtml(text);
	doc.convertUrlsToHtml();
	
	if(config_file.readBoolEntry("osdhints", "show_emoticons", false))
	{
		widgets.at(id)->mimeSourceFactory()->addFilePath(emoticons->themePath());
		emoticons->expandEmoticons(doc, bg_color);
	}
	text = doc.generateHtml();
	
	data.message = text;

	switch(maskEffect)
	{
		case 1:
			data.maskEffect = OSDWidget::Dissolve;
			break;
		default:
			data.maskEffect = OSDWidget::Plain;
			break;
	}
	
	data.fg_color = fg_color;
	data.bg_color = bg_color;
	data.border_color = border_color;
	data.duration = timeout;
	data.id = id;
	data.translucency_val = translucency_level;
	if (!senders.contains("Gadu", config_file.readEntry("General", "UIN")))
		data.users = senders;

	widgets.at(id)->setData(data);
	widgets.at(id)->prepare();
	
// TODO: zoptymalizowa? :)
	switch(corner)
	{
		case 0:		// Left top
			widgets.at(id)->setX(def_x);
			
			if ((widgets.count() > 1 && new_osd == 0) || new_osd != 0)
				height += widgets.at(id)->getHeight() + separator;
			break;
		case 1:		// Left bottom
			widgets.at(id)->setX(def_x);
			
			if ((widgets.count() > 1 && new_osd == 2) || new_osd != 2)
				height += widgets.at(id)->getHeight() + separator;
			break;
		case 2:		// Right top
			widgets.at(id)->setX(def_x - widgets.at(id)->getWidth());

			if ((widgets.count() > 1 && new_osd == 0) || new_osd != 0)
				height += widgets.at(id)->getHeight() + separator;
			break;
		case 3:		// Right bottom
			widgets.at(id)->setX(def_x - widgets.at(id)->getWidth());
			
			if ((widgets.count() > 1 && new_osd == 2) || new_osd != 2)
				height += widgets.at(id)->getHeight() + separator;
			break;
	}
	
	switch(new_osd)
	{
		case 0:
			widgets.at(id)->setY(def_y - height);
			break;
		case 1:
			if (widgets.count() > 1)
			{
				unsigned int shift;
				if (corner == 0 || corner == 2)
					shift = widgets.at(id)->getHeight() + separator;
				else
					shift = widgets.at(id - 1)->getHeight() + separator;
		
				for (unsigned int iter = 0; iter <= widgets.count() - 2; iter++)
				{
					widgets.at(iter)->setY(widgets.at(iter)->getY() + shift);
					widgets.at(iter)->QWidget::hide();		//na prawd nie rozumiem, po co :|
					widgets.at(iter)->display();
				}
			}
			if (corner == 1 || corner == 3)
				widgets.at(id)->setY(def_y - widgets.at(id)->getHeight());
			else
				widgets.at(id)->setY(def_y);
			break;
		case 2:
			widgets.at(id)->setY(def_y + height - widgets.at(id)->getHeight() - separator);
			break;
		case 3:
			if (widgets.count() > 1)
			{
				unsigned int shift;
				if (corner == 1 || corner == 3)
					shift = widgets.at(id)->getHeight() + separator;
				else
					shift = widgets.at(id - 1)->getHeight() + separator;
		
				for (unsigned int iter = 0; iter <= widgets.count() - 2; iter++)
				{
					widgets.at(iter)->setY(widgets.at(iter)->getY() - shift);
					widgets.at(iter)->QWidget::hide();		//na prawd nie rozumiem, po co :|
					widgets.at(iter)->display();
				}
			}
			if (corner == 1 || corner == 3)
				widgets.at(id)->setY(def_y - widgets.at(id)->getHeight());
			else
				widgets.at(id)->setY(def_y);
			break;
		case 4:
			widgets.at(id)->setY(def_y);
			break;
			
	}
	
	connect(widgets.at(id), SIGNAL(timeout(int, bool)), this, SLOT(timeout(int, bool)));
	connect(widgets.at(id), SIGNAL(deleteAll(void)), this, SLOT(deleteAll(void)));
	
	widgets.at(id)->display();
	
	kdebugf2();
}

void OSDManager::timeout(int id, bool chat)
{
	kdebugf();
	kdebugm(KDEBUG_INFO, "timeout: OSDWidget id: %d\n", id);

	
	UserListElements users = widgets.at(id)->getUsers();
	int shift = 0;
		
	int iter = 0;
	
	switch(new_osd)
	{
		case 0:
		{
			int new_id = 0;
			int count = widgets.count();
			while(iter < count)
			{
				if (widgets.at(iter)->getID() == id || (chat && widgets.at(iter)->getUsers() == users))
				{
					if (count > 1)
					{
						int a = widgets.at(iter)->getHeight() + separator;
						if (iter != 0 || corner == 1 || corner == 3)
						{
							height -= a;
							shift += a;
						}
					}
					else
					{
						shift = 0;
						height = 0;
					}
					
					widgets.at(iter)->close();
					widgets.remove(iter);
					count--;
				}
				else
				{
					if (iter == 0 && (corner == 0 || corner == 2))
					{
						shift = abs(def_y - widgets.at(iter)->getY());
						height -= def_y - widgets.at(iter)->getY();
					}
					if (shift != 0)
					{
						widgets.at(iter)->setY(widgets.at(iter)->getY() + shift);
						widgets.at(iter)->setID(new_id);
						widgets.at(iter)->QWidget::hide();
						widgets.at(iter)->display();
					}
					
					iter++;
					new_id++;
				}
			}
			break;
		}
		case 2:
		{
			int new_id = 0;
			int count = widgets.count();
			while(iter < count)
			{
				if (widgets.at(iter)->getID() == id || (chat && widgets.at(iter)->getUsers() == users))
				{
					if (count > 1)
					{
						int a = widgets.at(iter)->getHeight() + separator;
						if (iter != 0 || corner == 0 || corner == 2)
						{
							shift += a;
							height -= a;
						}
					}
					else
					{
						shift = 0;
						height = 0;
					}
					
					widgets.at(iter)->close();
					widgets.remove(iter);
					count--;
				}
				else
				{
					if (iter == 0 && (corner == 1 || corner == 3))
					{
						shift = abs(def_y - widgets.at(iter)->getY() - widgets.at(iter)->getHeight());
						height += def_y - widgets.at(iter)->getY() - widgets.at(iter)->getHeight();
					}
					if (shift != 0)
					{
						widgets.at(iter)->setY(widgets.at(iter)->getY() - shift);
						widgets.at(iter)->setID(new_id);
						widgets.at(iter)->QWidget::hide();
						widgets.at(iter)->display();
					}
					
					iter++;
					new_id++;
				}
			}
			break;
		}
		case 1:
		{
			OSDWidget *ptr = widgets.last();
			
			do
			{
				if (ptr->getID() == id || (chat && ptr->getUsers() == users))
				{
					shift += ptr->getHeight() + separator;
					
					ptr->close();
					widgets.remove(ptr);
					
					if (ptr->getID() == widgets.count())	//TODO: jako tak adniej :)
					{
						ptr = widgets.current();
					}
					else
						ptr = widgets.prev();
				}
				else
				{
					if (ptr == widgets.getLast() && (corner == 1 || corner == 3))
						shift = abs(def_y - ptr->getY() - ptr->getHeight());
					
					if (shift != 0)
					{
						ptr->setY(ptr->getY() - shift);
						ptr->QWidget::hide();
						ptr->display();
					}
					
					ptr = widgets.prev();
				}
			} while (ptr);
			
			
			int new_id = 0;
			for (ptr = widgets.first(); ptr; ptr = widgets.next())
			{
				ptr->setID(new_id);
				
				new_id++;
			}
			
			break;
		}
		case 3:
		{
			OSDWidget *ptr = widgets.last();
			
			do
			{
				if (ptr->getID() == id || (chat && ptr->getUsers() == users))
				{
					shift += ptr->getHeight() + separator;
					
					ptr->close();
					widgets.remove(ptr);
					
					if (ptr->getID() == widgets.count())	//TODO: jako tak adniej :)
					{
						ptr = widgets.current();
					}
					else
						ptr = widgets.prev();
				}
				else
				{
					if (ptr == widgets.getLast() && (corner == 0 || corner == 2))
						shift = abs(def_y - ptr->getY());
					
					if (shift != 0)
					{
						ptr->setY(ptr->getY() + shift);
						ptr->QWidget::hide();
						ptr->display();
					}
					
					ptr = widgets.prev();
				}
			} while (ptr);
			
			
			int new_id = 0;
			for (ptr = widgets.first(); ptr; ptr = widgets.next())
			{
				ptr->setID(new_id);
				
				new_id++;
			}
			
			break;
		}
	}

	kdebugf2();
}

void OSDManager::deleteAll(void)
{
	kdebugf();
	
	height = 0;

	CONST_FOREACH(it, widgets)
	{
		(*it)->close();
	}
	widgets.clear();

	kdebugf2();
}

void OSDManager::windowActivationChanged(bool b, const UserGroup *users)
{
	kdebugf();
	
	if (!b)
	{
		kdebugm(KDEBUG_INFO, "windowActivated\n");
		
		UserListElements u_list = users->toUserListElements();
		CONST_FOREACH(i, widgets)
		{
			if ((*i)->getUsers() == u_list)
			{
				timeout((*i)->getID(), true);
				
				kdebugf2();
				return;
			}
			
		}
	}

	kdebugf2();
}

void OSDManager::addMessage(const UserListElements senders, const QString &msg, const QString prefix)
{
	kdebugf();

	QFont font = config_file.readFontEntry("osdhints", prefix + "font");
	QColor fg_color = config_file.readColorEntry("osdhints", prefix + "fgcolor");
	QColor bg_color = config_file.readColorEntry("osdhints", prefix + "bgcolor");
	QColor border_color = config_file.readColorEntry("osdhints", prefix + "bordercolor");
	unsigned int timeout = config_file.readUnsignedNumEntry("osdhints", prefix + "timeout", 3600);
	QString syntax = config_file.readEntry("osdhints", prefix + "syntax", "run config dialog!");
	int maskEffect = config_file.readNumEntry("osdhints", prefix + "mask_effect");
	
// cytowanie
	QString text;
	unsigned int citeSign = config_file.readUnsignedNumEntry("osdhints","CiteSign", 50);
		
	if (msg.length() <= citeSign)
		text = msg;
	else
		text = msg.left(citeSign);

// skadnia
	syntax = KaduParser::parse(syntax, senders[0], true);
	text = syntax.replace("%&m", msg);

	addOSD(text, font, fg_color, bg_color, border_color, timeout, maskEffect, senders);
	
	kdebugf2();
}

void OSDManager::addStatus(const UserListElements senders, const QString prefix)
{
	kdebugf();
	
	QFont font = config_file.readFontEntry("osdhints", prefix + "font");
	QColor fg_color = config_file.readColorEntry("osdhints", prefix + "fgcolor");
	QColor bg_color = config_file.readColorEntry("osdhints", prefix + "bgcolor");
	QColor border_color = config_file.readColorEntry("osdhints", prefix + "bordercolor");
	unsigned int timeout = config_file.readUnsignedNumEntry("osdhints", prefix + "timeout", 3600);
	QString syntax = config_file.readEntry("osdhints", prefix + "syntax", "run config dialog!");
	int maskEffect = config_file.readNumEntry("osdhints", prefix + "mask_effect");

// skadnia
	syntax = KaduParser::parse(syntax, senders[0], true);

	addOSD(syntax, font, fg_color, bg_color, border_color, timeout, maskEffect, senders);
	
	kdebugf2();
}

/**

	SLOTY dla notify

**/

void OSDManager::newChat(Protocol * /*protocol*/, UserListElements senders, const QString &msg, time_t /*t*/)
{
	kdebugf();
	
	addMessage(senders, msg, "OSDNewChat_");

	kdebugf2();
}

void OSDManager::newMessage(Protocol * /*protocol*/, UserListElements senders, const QString &msg, time_t /*t*/, bool &/*grab*/)
{
	kdebugf();
	
	Chat *chat = chat_manager->findChat(senders);
	if (chat->isActiveWindow() || chat == NULL)
	{
		kdebugmf(KDEBUG_INFO|KDEBUG_FUNCTION_END, "end: chat->isActiveWindow() || chat == NULL\n");
		return;
	}
	
	connect(chat, SIGNAL(windowActivationChanged(bool, const UserGroup *)), this, SLOT(windowActivationChanged(bool, const UserGroup *)));
	//TODO: czy trzeba disconnect?
	
	addMessage(senders, msg, "OSDNewMessage_");

	kdebugf2();
}

void OSDManager::userChangedStatusToAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();
	
	UserListElements ulist;
//	if (config_file.readBoolEntry("osdhints", "OpenChatOnClick", false))
		ulist.append(ule);
	
	QString prefix;
	if (ule.status(protocolName).hasDescription())
		prefix = "OSDOnlineD_";
	else
		prefix = "OSDOnline_";
	
	addStatus(ulist, prefix);

	kdebugf2();
}

void OSDManager::userChangedStatusToBusy(const QString &protocolName, UserListElement ule)
{
	kdebugf();
	
	UserListElements ulist;
//	if (config_file.readBoolEntry("osdhints", "OpenChatOnClick", false))
		ulist.append(ule);
	
	QString prefix;
	if (ule.status(protocolName).hasDescription())
		prefix = "OSDBusyD_";
	else
		prefix = "OSDBusy_";
	
	addStatus(ulist, prefix);

	kdebugf2();
}

void OSDManager::userChangedStatusToInvisible(const QString &protocolName, UserListElement ule)
{
	kdebugf();
	
	UserListElements ulist;
//	if (config_file.readBoolEntry("osdhints", "OpenChatOnClick", false))
		ulist.append(ule);
	
	QString prefix;
	if (ule.status(protocolName).hasDescription())
		prefix = "OSDInvisibleD_";
	else
		prefix = "OSDInvisible_";
	
	addStatus(ulist, prefix);

	kdebugf2();
}

void OSDManager::userChangedStatusToNotAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();
	
	UserListElements ulist;
//	if (config_file.readBoolEntry("osdhints", "OpenChatOnClick", false))
		ulist.append(ule);
	
	QString prefix;
	if (ule.status(protocolName).hasDescription())
		prefix = "OSDOfflineD_";
	else
		prefix = "OSDOffline_";
	
	addStatus(ulist, prefix);

	kdebugf2();
}

void OSDManager::connectionError(Protocol *, const QString &message)
{
	kdebugf();
	
	UserListElements ulist;

	QFont font = config_file.readFontEntry("osdhints", "OSDError_font");
	QColor fg_color = config_file.readColorEntry("osdhints", "OSDError_fgcolor");
	QColor bg_color = config_file.readColorEntry("osdhints", "OSDError_bgcolor");
	QColor border_color = config_file.readColorEntry("osdhints", "OSDError_bordercolor");
	unsigned int timeout = config_file.readUnsignedNumEntry("osdhints", "OSDError_timeout", 3600);
	QString syntax = config_file.readEntry("osdhints", "OSDError_syntax", "run config dialog!");
	int maskEffect = config_file.readNumEntry("osdhints", "OSDError_mask_effect");

// skadnia
//	syntax = KaduParser::parse(syntax, senders[0], true);
	syntax = syntax.replace("%&m", message);
	addOSD(syntax, font, fg_color, bg_color, border_color, timeout, maskEffect, ulist);
	
	kdebugf2();
}

void OSDManager::userBoxChangeToolTip(const QPoint &point, UserListElement user, bool show)
{
	kdebugf();
	kdebugm(KDEBUG_INFO, "user: '%s', x:%d, y:%d\n", user.altNick().local8Bit().data(), show, point.x(), point.y());

	if (show)
	{
		if (tipWidget != NULL)
			return;
		
		tipWidget = new OSDWidget(0);
		
		tipWidget->setFont(config_file.readFontEntry("osdhints", "OSDToolTip_font"));
		
		OSDWidget::OSDData data;
		
		data.fg_color = config_file.readColorEntry("osdhints", "OSDToolTip_fgcolor");
		data.bg_color = config_file.readColorEntry("osdhints", "OSDToolTip_bgcolor");
		data.border_color = config_file.readColorEntry("osdhints", "OSDToolTip_bordercolor");
		data.duration = 0;
		switch(config_file.readNumEntry("osdhints", "OSDToolTip_mask_effect"))
		{
			case 1:
				data.maskEffect = OSDWidget::Dissolve;
				break;
			default:
				data.maskEffect = OSDWidget::Plain;
				break;
		}

		data.id = -1;
		data.translucency_val = translucency_level;
		
		QPoint pos(kadu->userbox()->mapToGlobal(point) + QPoint(5, 5));

		data.x = pos.x();
		data.y = pos.y();
		
		QString text = KaduParser::parse(config_file.readEntry("osdhints", "OSDToolTip_syntax", "run config dialog!"), user, true);
		
		while (text.endsWith("<br>"))
			text.setLength(text.length() - 4 /* 4 == QString("<br>").length()*/);
		while (text.startsWith("<br>"))
			text = text.right(text.length() - 4 /* 4 == QString("<br>").length()*/);

		
		HtmlDocument doc;
		doc.parseHtml(text);
		doc.convertUrlsToHtml();
	
		if(config_file.readBoolEntry("osdhints", "show_emoticons", false))
		{
			tipWidget->mimeSourceFactory()->addFilePath(emoticons->themePath());
			emoticons->expandEmoticons(doc, data.bg_color);
		}
		text = doc.generateHtml();
		data.message = text;
		
		tipWidget->setData(data);
		tipWidget->prepare();
		
		// czy nie zasania czego; czy nie wychodzi za pulpit jest i tak sprawdzane w prepare()
		QSize desktopSize = QApplication::desktop()->size();
		if (pos.x() + tipWidget->getWidth() >= desktopSize.width())
			pos.setX(pos.x() - tipWidget->getWidth() - 10);
		if (pos.y() + tipWidget->getHeight() >= desktopSize.height())
			pos.setY(pos.y() - tipWidget->getHeight() - 10);
		
		tipWidget->setX(pos.x());
		tipWidget->setY(pos.y());
		
		tipWidget->display();
	}
	else
	{
		delete tipWidget;
		tipWidget = NULL;
	}
	
	kdebugf2();
}

void OSDManager::message(const QString &from, const QString &msg, const QMap<QString, QVariant> *parameters, const UserListElement *ule)
{
	kdebugf();
	
	// TODO: mime dla pixmap...
	
	UserListElements userlist;
	if (ule != NULL)
		userlist.append(*ule);
	
	QString syntax = config_file.readEntry("osdhints", "OSDMessage_syntax", "run config dialog!");
	syntax = KaduParser::parse(syntax, *ule, true);
	syntax.replace("%&m", msg);
	
	QColor border_color = config_file.readColorEntry("osdhints", "OSDMessage_bordercolor");
	int maskEffect = config_file.readNumEntry("osdhints", "OSDMessage_mask_effect");
	
	QString image;
//	QPixmap pixmap;
	QFont font;
	QString fg_color;
	QString bg_color;
	unsigned int timeout;
	bool ok;


	if (parameters!=NULL)
	{
		
//		pixmap=(*parameters)["Pixmap"].toPixmap();
		font=(*parameters)["Font"].toFont();
		fg_color=(*parameters)["Foreground color"].toString();
		bg_color=(*parameters)["Background color"].toString();
		timeout=(*parameters)["Timeout"].toUInt(&ok) * 1000; //1000, bo milisekundy
	}
	
	if (fg_color == "")
		fg_color = config_file.readEntry("osdhints", "OSDMessage_fgcolor");
	if (bg_color == "")
		bg_color = config_file.readEntry("osdhints", "OSDMessage_bgcolor");


	
//	if (pixmap.isNull())
//		image=config_file.readEntry("osdhints", "OSDMessage_icon");
	if (font==qApp->font())
		font=config_file.readFontEntry("osdhints", "OSDMessage_font");
	if (!ok)
		timeout=config_file.readUnsignedNumEntry("osdhints", "OSDMessage_timeout", 3000);

	if (!from.isEmpty())
		syntax.replace("%&f", tr("From: ") + from + "<br>");
	else
		syntax.replace("%&f", "");

	addOSD(syntax, font, fg_color, bg_color, border_color, timeout, maskEffect, userlist);

	kdebugf2();
}


void OSDManager::externalEvent(const QString &notifyType, const QString &msg, const UserListElements &ules)
{
	kdebugf();

	if (ules.count() > 0)
		message("", msg, 0, &ules[0]);
	else
		message("", msg, 0, 0);

	kdebugf2();
}
