/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: wincshf.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:51:40 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <prewin.h>
#include "shutil.h"
#include <postwin.h>

#include "winshell.hxx"


//****************************************************************************
//	CShellFolderData
//****************************************************************************

//----------------------------------------------------------------------------
// Initialize Data for CShellFolder
//----------------------------------------------------------------------------

CShellFolderData::CShellFolderData() :
m_pShellFolder( NULL ),
m_pEnumIDList( NULL ),
m_pShellIcon( NULL ),
m_bIsOpen( FALSE ),
m_dwContentFlags( SHCONTF_FOLDERS | SHCONTF_NONFOLDERS | SHCONTF_INCLUDEHIDDEN ),
m_hCancelEvent( INVALID_HANDLE_VALUE )
{
}

CShellFolderData::~CShellFolderData()
{
	if ( m_pEnumIDList )
		m_pEnumIDList->Release();
	if ( m_pShellIcon )
		m_pShellIcon->Release();
	if ( m_pShellFolder )
		m_pShellFolder->Release();

	if ( INVALID_HANDLE_VALUE != m_hCancelEvent )
	{
		PulseEvent( m_hCancelEvent );
		CloseHandle( m_hCancelEvent );
	}
}

//****************************************************************************
//	CShellFolder
//****************************************************************************

//----------------------------------------------------------------------------
// Default initialisation
//----------------------------------------------------------------------------

void CShellFolder::Initialize( IShellFolder *pShellFolder )
{
	m_pShellFolder = pShellFolder;

	if ( m_pShellFolder )
	{
		m_pShellFolder->AddRef();
		if ( NOERROR == m_pShellFolder->EnumObjects( NULL, m_dwContentFlags, &m_pEnumIDList ) )
		{
			m_bIsOpen = TRUE;
			m_hCancelEvent = CreateEvent( NULL, TRUE, FALSE, NULL );

			m_pShellFolder->QueryInterface( IID_IShellIcon, (LPVOID *)&m_pShellIcon );
		}
	}
}

//----------------------------------------------------------------------------

void CShellFolder::Initialize( LPCITEMIDLIST pidl )
{
	IShellFolder *pFolder;

	if ( WIN_SHGetFolderFromIDList( pidl, &pFolder ) )
	{
		Initialize( pFolder );
		pFolder->Release();
	}
}

//----------------------------------------------------------------------------
// Constructors
//----------------------------------------------------------------------------

CShellFolder::CShellFolder( const CShellFolder &rShellFolder ) :
CShellFolderData()
{
	Initialize( rShellFolder.m_pShellFolder );
}

//----------------------------------------------------------------------------

CShellFolder::CShellFolder( IShellFolder *pShellFolder ) :
CShellFolderData()
{
	Initialize( pShellFolder );
}

//----------------------------------------------------------------------------

CShellFolder::CShellFolder( const CItemIDList & rIDList ) :
CShellFolderData()
{
	Initialize( rIDList );
}

//----------------------------------------------------------------------------
// Reset ( virtual )
//----------------------------------------------------------------------------

BOOL CShellFolder::Reset()
{
	BOOL	bSuccess;

	if ( m_pEnumIDList )
	{
		m_pEnumIDList->Release();
		m_pEnumIDList = NULL;
	}

	bSuccess = ( NOERROR == m_pShellFolder->EnumObjects( NULL, m_dwContentFlags, &m_pEnumIDList ) );


	return bSuccess;
}

//----------------------------------------------------------------------------
// GetNextValidID
//----------------------------------------------------------------------------

BOOL CShellFolder::GetNextValidID( CItemIDList & rID )
{
	BOOL	bFound, bValid = FALSE;

	do
	{
		bFound = GetNextID( rID );
		if ( bFound )
			bValid = ValidateID( rID );

	} while ( bFound && !bValid );

	return bValid;
}

//----------------------------------------------------------------------------
// GetNextID ( virtual )
//----------------------------------------------------------------------------

BOOL CShellFolder::GetNextID( CItemIDList & rID )
{
	BOOL	bFound = FALSE;

	if ( m_pEnumIDList )
	{
		ULONG			nFetched;
		HRESULT			hResult;
		LPITEMIDLIST	pidl;

		hResult = m_pEnumIDList->Next( 1, &pidl, &nFetched );

		if ( NOERROR == hResult )
		{
			rID = CItemIDList(pidl);
			WIN_SHFree( pidl );

			bFound = TRUE;
		}

	}

	return bFound;
}

//----------------------------------------------------------------------------
// ValidateID (virtual)
//----------------------------------------------------------------------------

BOOL CShellFolder::ValidateID( const CItemIDList & rID )
{
	return TRUE;
}

//----------------------------------------------------------------------------
// GetAttributesOf
//----------------------------------------------------------------------------

BOOL CShellFolder::GetAttributesOf( const CItemIDList & rIDList, LPDWORD pdwInOut )
{
	BOOL	bSuccess = FALSE;

	if ( m_pShellFolder )
	{
		HRESULT			hResult;
		LPCITEMIDLIST	pidl = rIDList;

		if ( rIDList.GetTokenCount() > 1 )
		{
			CItemIDList	aFolderID = rIDList[0];

			IShellFolder	*pSubFolder;
			hResult = m_pShellFolder->BindToObject( aFolderID, NULL, IID_IShellFolder, (LPVOID *)&pSubFolder );

			if ( NOERROR == hResult )
			{
				pidl = (LPCITEMIDLIST)((LPBYTE)pidl + pidl->mkid.cb);

				CItemIDList aSubIDList( pidl );
				CShellFolder	aSubFolder( pSubFolder );
				pSubFolder->Release();

				return aSubFolder.GetAttributesOf( aSubIDList, pdwInOut );
			}
		}
		else
		{
			hResult = m_pShellFolder->GetAttributesOf( 1, &pidl, pdwInOut );
			bSuccess = SUCCEEDED( hResult );
		}
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// GetNameOf
//----------------------------------------------------------------------------

BOOL CShellFolder::GetNameOf( const CItemIDList & rIDList, String & rName )
{
	BOOL	bSuccess = FALSE;

	if ( m_pShellFolder )
	{
		HRESULT			hResult;
		LPCITEMIDLIST	pidl = rIDList;
		STRRET			strName;

		hResult = m_pShellFolder->GetDisplayNameOf( pidl, SHGDN_INFOLDER, &strName );
		bSuccess = SUCCEEDED( hResult );

		if ( bSuccess )
		{
			TCHAR	szDisplayName[MAX_PATH] = "";

			WIN_SHStrRetToMultiByte( pidl, &strName, szDisplayName, MAX_PATH );
			rName = szDisplayName;
		}
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// GetFileInfo
//----------------------------------------------------------------------------

BOOL CShellFolder::GetFileInfo( const CItemIDList & rIDList, WIN32_FIND_DATA *pFileData )
{
	BOOL	bSuccess = FALSE;

	if ( m_pShellFolder )
	{
		LPCITEMIDLIST	pidl = rIDList;
		HRESULT			hResult;

		hResult = WIN_SHGetDataFromIDList( m_pShellFolder, pidl, SHGDFIL_FINDDATA, pFileData, sizeof(WIN32_FIND_DATA) );

		if ( NOERROR == hResult )
		{
			bSuccess = TRUE;
		}
		else
		{
			STRRET			strName;

			hResult = m_pShellFolder->GetDisplayNameOf( pidl, SHGDN_NORMAL | SHGDN_FORPARSING, &strName );
			bSuccess = SUCCEEDED( hResult );

			if ( bSuccess )
			{
				TCHAR	szPath[MAX_PATH] = "";

				WIN_SHStrRetToMultiByte( pidl, &strName, szPath, MAX_PATH );
				HANDLE	 hFind = FindFirstFile( szPath, pFileData );
				if ( hFind != INVALID_HANDLE_VALUE )
				{
					FindClose( hFind );
					bSuccess = TRUE;
				}
			}
		}
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// GetVolumeInfo
//----------------------------------------------------------------------------

BOOL CShellFolder::GetVolumeInfo( const CItemIDList & rIDList, WIN32_VOLUME_DATA *pVolumeData )
{
	BOOL	bSuccess = FALSE;

	if ( m_pShellFolder )
	{
		HRESULT			hResult;
		LPCITEMIDLIST	pidl = rIDList;
		STRRET			strName;

		hResult = m_pShellFolder->GetDisplayNameOf( pidl, SHGDN_NORMAL | SHGDN_FORPARSING, &strName );
		bSuccess = SUCCEEDED( hResult );

		if ( bSuccess )
		{
			TCHAR	szPath[MAX_PATH] = "";

			WIN_SHStrRetToMultiByte( pidl, &strName, szPath, MAX_PATH );

			lstrcpy( pVolumeData->cDeviceName, szPath );

			pVolumeData->cDeviceName[3] = '\0';
			pVolumeData->cVolumeName[0] = '\0';
			pVolumeData->cFileSystemName[0] = '\0';
			pVolumeData->dwSerialNumber = 0;
			pVolumeData->dwFileSystemFlags = 0;
			pVolumeData->nMaxComponentLength = MAX_PATH;

			bSuccess = GetVolumeInformation( 
				pVolumeData->cDeviceName, 
				pVolumeData->cVolumeName, MAX_PATH, 
				&pVolumeData->dwSerialNumber, 
				&pVolumeData->nMaxComponentLength, 
				&pVolumeData->dwFileSystemFlags, 
				pVolumeData->cFileSystemName, MAX_PATH );
		}
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// SetNameOf
//----------------------------------------------------------------------------

BOOL CShellFolder::SetNameOf( const CItemIDList & rIDList, const String & rName, CItemIDList & rNewIDList )
{
	BOOL	bSuccess = FALSE;

	if ( m_pShellFolder )
	{
		HRESULT			hResult;
		LPCITEMIDLIST	pidl = rIDList;
		LPITEMIDLIST	pidlOut = NULL;
		WCHAR			wszName[MAX_PATH];

		MultiByteToWideChar( CP_ACP, 0, rName.GetStr(), -1, wszName, MAX_PATH );

		hResult = m_pShellFolder->SetNameOf( NULL, pidl, wszName, SHGDN_NORMAL, &pidlOut );
		bSuccess = SUCCEEDED( hResult );

		if ( bSuccess )
		{
			rNewIDList = pidlOut;
			WIN_SHFree( pidlOut );
		}
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// DeleteItem
//----------------------------------------------------------------------------

BOOL CShellFolder::DeleteItem( const CItemIDList & rIDList )
{
	BOOL			fSuccess = FALSE;
	LPCITEMIDLIST	pidl = rIDList;

	if ( m_pShellFolder )
	{
		IContextMenu	*pContextMenu;

		if ( NOERROR == m_pShellFolder->GetUIObjectOf( NULL, 1, &pidl, IID_IContextMenu, NULL, (LPVOID *)&pContextMenu ) )
		{
			HMENU	hMenu = CreatePopupMenu();

			if ( hMenu )
			{
				HRESULT hResult = pContextMenu->QueryContextMenu( hMenu, 0, 0, 1000, CMF_NORMAL | CMF_CANRENAME | CMF_INCLUDESTATIC );

				if ( SUCCEEDED(hResult) )
				{
					CMINVOKECOMMANDINFO	ici;

					ici.cbSize = sizeof(ici);
					ici.fMask = 0;
					ici.hwnd = NULL;
					ici.lpParameters = NULL;
					ici.lpDirectory = NULL;
					ici.nShow = SW_SHOWNORMAL;
					ici.dwHotKey = 0;
					ici.hIcon = NULL;
					ici.lpVerb = CMDSTR_DELETE;

					fSuccess = (BOOL)(NOERROR == pContextMenu->InvokeCommand( &ici ));
				}

				DestroyMenu( hMenu );
			}


			pContextMenu->Release();
		}
	}

	return fSuccess;
}

//----------------------------------------------------------------------------
// CompareIDs (virtual)
//----------------------------------------------------------------------------

int CShellFolder::CompareIDs( const CItemIDList & rID1, const CItemIDList & rID2 )
{
	if ( m_pShellFolder )
		return m_pShellFolder->CompareIDs( 0, rID1, rID2 );
	else
		return -1;
}

//----------------------------------------------------------------------------
// GetContextMenu (virtual)
//----------------------------------------------------------------------------

IContextMenu *CShellFolder::GetContextMenu( int nItems, const CItemIDList *rID )
{
	IContextMenu	*pContextMenu = NULL;

	if ( m_pShellFolder )
	{
		LPCITEMIDLIST	*pidlList = new LPCITEMIDLIST[nItems];


		if ( pidlList )
		{
			for ( int i = 0; i < nItems; i++ )
				pidlList[i] = rID[i];
				
			m_pShellFolder->GetUIObjectOf( NULL, 1, pidlList, IID_IContextMenu, NULL, (LPVOID *)&pContextMenu );

			delete [nItems] pidlList;
		}
	}

	return pContextMenu;
}

//----------------------------------------------------------------------------
// GetIconLocation (virtual)
//----------------------------------------------------------------------------

String CShellFolder::GetIconLocation( const CItemIDList & rID )
{
	HRESULT	hResult;
	LPCITEMIDLIST	pidl = rID;
	BOOL	bFound = FALSE;;
	String	aLocation;


	if ( !bFound && m_pShellIcon )
	{
		INT		iIndex = 0;

		hResult = m_pShellIcon->GetIconOf( pidl, GIL_FORSHELL, &iIndex );
		if ( NOERROR == hResult )
		{
			aLocation = String("#") + String( iIndex );
			bFound = TRUE;
		}
	}

	if ( !bFound && m_pShellFolder )
	{
		IExtractIcon	*pExtractIcon;

		hResult = m_pShellFolder->GetUIObjectOf( NULL, 1, &pidl, IID_IExtractIcon, NULL, (LPVOID *)&pExtractIcon );
		if ( NOERROR == hResult )
		{
			int		iIndex = 0;
			UINT	dwFlags = 0;
			TCHAR	szIconFile[MAX_PATH];

			hResult = pExtractIcon->GetIconLocation( GIL_FORSHELL, szIconFile, MAX_PATH, &iIndex, &dwFlags );
			if ( NOERROR == hResult )
			{
				aLocation = String(">") + String(szIconFile) + String(",") + String(iIndex);
				bFound = TRUE;
			}

			pExtractIcon->Release();
		}
	}
	
	return aLocation;
}

//----------------------------------------------------------------------------
// WaitForNotification (virtual)
//----------------------------------------------------------------------------

NotificationEvent CShellFolder::WaitForChanges()
{
	return NotificationEvent_Error;
}

//----------------------------------------------------------------------------
// CancelWaitNotifications (virtual)
//----------------------------------------------------------------------------

void CShellFolder::CancelWaitNotifications()
{
	PulseEvent( m_hCancelEvent );
}

//****************************************************************************
//	CFileSystemFolder
//****************************************************************************

//----------------------------------------------------------------------------
// Default initialisation
//----------------------------------------------------------------------------

void CFileSystemFolder::Initialize( LPCSTR pszPath )
{
	IShellFolder	*pFolder;

	if ( WIN_SHGetFolderFromPath( pszPath, &pFolder ) )
	{
		CShellFolder::Initialize( pFolder );
		strcpy( m_szPath, pszPath );
	}
	else
		m_szPath[0] = '\0';
}

//----------------------------------------------------------------------------
// Constructor
//----------------------------------------------------------------------------

CFileSystemFolder::CFileSystemFolder( LPCTSTR pszPath ) :
CShellFolder()
{
	Initialize( pszPath );
}

//----------------------------------------------------------------------------
// WaitForNotification (virtual)
//----------------------------------------------------------------------------

NotificationEvent CFileSystemFolder::WaitForChanges()
{
	NotificationEvent	eEvent = NotificationEvent_Error;

	// Handelt es sich um einen reinen FileSystem-Ordner, so nutzen wird die Notification 

	HANDLE	hObjects[2];
	
	hObjects[0] = m_hCancelEvent;
	hObjects[1] = FindFirstChangeNotification( m_szPath, FALSE, FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME | FILE_NOTIFY_CHANGE_LAST_WRITE );

	if ( hObjects[1] != INVALID_HANDLE_VALUE  )
	{
		DWORD dwEvent = WaitForMultipleObjects( sizeof(hObjects) / sizeof(HANDLE), hObjects, FALSE, INFINITE );
		
		switch ( dwEvent )
		{
		case WAIT_OBJECT_0:
			eEvent = NotificationEvent_Canceled;
			break;
		case WAIT_OBJECT_0 + 1:
			eEvent = NotificationEvent_Signaled;
			break;
		default:
			break;
		}

		FindCloseChangeNotification( hObjects[1] );
	}

	return eEvent;
}

//****************************************************************************
//	CSpecialFolder
//****************************************************************************

//----------------------------------------------------------------------------
// Default initialisation
//----------------------------------------------------------------------------

void CSpecialFolder::Initialize( int nFolder )
{
	IShellFolder	*pFolder;

	if ( WIN_SHGetSpecialFolder( nFolder, &pFolder ) )
	{
		CShellFolder::Initialize( pFolder );
		m_nFolder = nFolder;
	}
	else
		m_nFolder = -1;
}

//----------------------------------------------------------------------------
// Constructor
//----------------------------------------------------------------------------

CSpecialFolder::CSpecialFolder( int nFolder ) :
CShellFolder()
{
	Initialize( nFolder );
}

//****************************************************************************
//	CVolumesFolder
//****************************************************************************

#define SHGII_CONTAINER_MASK		0x70

#define SHGII_COMPUTER				0x20

#define SHGII_COMPUTER_REMOVABLE	0x22
#define SHGII_COMPUTER_FIXED		0x23
#define SHGII_COMPUTER_REMOTE		0x24
#define SHGII_COMPUTER_CDROM		0x25
#define SHGII_COMPUTER_RAMDISK		0x26
#define SHGII_COMPUTER_FLOPPY525	0x28
#define SHGII_COMPUTER_FLOPPY35		0x29
#define SHGII_COMPUTER_NETWORK		0x2A
#define SHGII_COMPUTER_REGITEM		0x2E

#define SHGII_ROOT					0x10
#define SHGII_ROOT_REGITEM			0x1F

//----------------------------------------------------------------------------
// ValidateID (virtual)
//----------------------------------------------------------------------------

BOOL CVolumesFolder::ValidateID( const CItemIDList & rIDList )
{
	LPCITEMIDLIST	pidl = rIDList;

	return pidl->mkid.abID[0] < SHGII_COMPUTER_REGITEM;
}

//----------------------------------------------------------------------------
// WaitForChanges (virtual)
//----------------------------------------------------------------------------

NotificationEvent CVolumesFolder::WaitForChanges()
{
	return NotificationEvent_Error;
}

//****************************************************************************
// CWorkplaceFolder
//****************************************************************************

//----------------------------------------------------------------------------
// ValidateID (virtual)
//----------------------------------------------------------------------------

BOOL CWorkplaceFolder::ValidateID( const CItemIDList & rIDList )
{
	LPCITEMIDLIST	pidl = rIDList;

	return pidl->mkid.abID[0] >= SHGII_COMPUTER_REGITEM;
}

//----------------------------------------------------------------------------
// WaitForChanges (virtual)
//----------------------------------------------------------------------------

NotificationEvent CWorkplaceFolder::WaitForChanges()
{
	return NotificationEvent_Error;
}


//****************************************************************************
// CDesktopAncestorsFolder
//****************************************************************************

//----------------------------------------------------------------------------
// ValidateID (virtual)
//----------------------------------------------------------------------------

BOOL CDesktopAncestorsFolder::ValidateID( const CItemIDList & rIDList )
{
	BOOL	bSuccess = FALSE;
	LPCITEMIDLIST	pidl = rIDList;

	if ( pidl->mkid.abID[0] == SHGII_ROOT_REGITEM )
	{
		LPCLSID	pClsId = (LPCLSID)&pidl->mkid.abID[2];

		bSuccess =	memcmp( pClsId, &CLSID_MyComputer, sizeof(GUID) ) == 0 ||
					memcmp( pClsId, &CLSID_Network, sizeof(GUID) ) == 0;
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// WaitForChanges (virtual)
//----------------------------------------------------------------------------

NotificationEvent CDesktopAncestorsFolder::WaitForChanges()
{
	return NotificationEvent_Error;
}


//****************************************************************************
// CDesktopContensFolder
//****************************************************************************

//----------------------------------------------------------------------------
// ValidateID (virtual)
//----------------------------------------------------------------------------

BOOL CDesktopContentsFolder::ValidateID( const CItemIDList & rIDList )
{
	BOOL	bSuccess = TRUE;
	LPCITEMIDLIST	pidl = rIDList;

	if ( pidl->mkid.abID[0] == SHGII_ROOT_REGITEM )
	{
		LPCLSID	pClsId = (LPCLSID)&pidl->mkid.abID[2];

		bSuccess =	memcmp( pClsId, &CLSID_MyComputer, sizeof(GUID) ) != 0 &&
					memcmp( pClsId, &CLSID_Network, sizeof(GUID) ) != 0 &&
					memcmp( pClsId, &CLSID_RecycleBin, sizeof(GUID) ) != 0;
	}

	return bSuccess;
}

//----------------------------------------------------------------------------
// WaitForChanges (virtual)
//----------------------------------------------------------------------------

NotificationEvent CDesktopContentsFolder::WaitForChanges()
{
	NotificationEvent	eEvent = NotificationEvent_Error;

	// Filesystem-Notification fr Common-Desktop und User-Desktop
	// Der Rest interessiert uns erstmal nicht

	TCHAR	szDesktop[MAX_PATH] = "";
	TCHAR	szCommonDesktop[MAX_PATH] = "";

	HANDLE	hObjects[3];
	int		nObjects = 1;
	
	hObjects[0] = m_hCancelEvent;

	WIN_SHGetSpecialFolderPath( CSIDL_DESKTOPDIRECTORY, szDesktop );
	WIN_SHGetSpecialFolderPath( CSIDL_COMMON_DESKTOPDIRECTORY, szCommonDesktop );

	hObjects[nObjects] = FindFirstChangeNotification( szDesktop, FALSE, FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME | FILE_NOTIFY_CHANGE_LAST_WRITE );

	if ( hObjects[nObjects] != INVALID_HANDLE_VALUE )
		nObjects++;

	hObjects[nObjects] = FindFirstChangeNotification( szCommonDesktop, FALSE, FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME | FILE_NOTIFY_CHANGE_LAST_WRITE );

	if ( hObjects[nObjects] != INVALID_HANDLE_VALUE )
		nObjects++;

	if ( nObjects > 1 )
	{
		DWORD dwEvent = WaitForMultipleObjects( nObjects, hObjects, FALSE, INFINITE );
		
		switch ( dwEvent )
		{
		case WAIT_OBJECT_0:
			eEvent = NotificationEvent_Canceled;
			break;
		default:
			eEvent = NotificationEvent_Signaled;
			break;
		}

		if ( hObjects[1] != INVALID_HANDLE_VALUE )
			FindCloseChangeNotification( hObjects[1] );
		if ( hObjects[2] != INVALID_HANDLE_VALUE )
			FindCloseChangeNotification( hObjects[2] );
	}

	return eEvent;
}

