/*************************************************************************
*
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: pasfsound.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 12:59:23 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
************************************************************************/

#include <salsound.h>
#include <salimpsound.hxx>

#ifdef USE_PASF

#define _SV_PASFSOUND_CXX

// Replacement SalSound Implementation that makes use of PortAudio to abstract away
// hardware differences, and LibSndFile to convert and access sound formats
// This implementation should work with little or no modification for Linux
// SGI, Solaris, MacOSX, Linux (all arches), FreeBSD, and even WIN

#include <saldata.hxx>
#include <salinst.hxx>


#include <stdio.h>

using namespace vcl_sal;
using namespace vos;
using namespace osl;
using namespace rtl;

extern "C"
{
    
    /* callback that takes data from sound file and converts it and loads it */
    
    static int pasf_in_out_callback (void *inputBuffer, void * outputBuffer,
    unsigned long framesPerBuffer, PaTimestamp outTime, void * userData )
    {	
        PASFAudioData*	paudio_data;
        int		read_count, bufoff, bytecnt; 
        float*          buffer  = (float *) outputBuffer;
        
        /* prevent warnings for unused variables */
        inputBuffer = inputBuffer;
        outTime = outTime;
        
        /* get all other info needed from audio_data structure passed in as userData */
        paudio_data = (PASFAudioData*) userData;
        
        read_count = sf_readf_float (paudio_data->sndfile, buffer, framesPerBuffer);
        if (read_count < framesPerBuffer) 
        {	
	        /* fill in the remainder of the buffer with zeros */
            /* convert read_count of frames to offset into float* buffer */
	        bufoff = read_count * paudio_data->sfinfo.channels;
            bytecnt = (framesPerBuffer - read_count) * paudio_data->sfinfo.channels * sizeof(float);
            memset (&(buffer [read_count]), 0, bytecnt) ;
        }
        
        /* now check if we reached end of file or preset stopping point */
        if ((read_count == 0) || ((paudio_data->stop_frame > 0) 
            && (paudio_data->frame_count > paudio_data->stop_frame))) 
        {
            if (paudio_data->loop_count > 0) paudio_data->loop_count--;
            
            /* if need to loop again, seek back to start frame and reset frame count */
            if ((paudio_data->loop_count > 0) || (paudio_data->loop_count == -1)) 
            {
                sf_seek(paudio_data->sndfile, paudio_data->start_frame, SEEK_SET);
                paudio_data->frame_count = paudio_data->start_frame;
            } else {
                paudio_data->done_playing = 1 ;
            }
        } 
        
        paudio_data->frame_count = paudio_data->frame_count + framesPerBuffer;
        return paNoError;
    } 
}



PASFSound::PASFSound( ::X11SalSound* pSound) : 
VSound( pSound ), 
m_pData( NULL ),
m_pStream( NULL ) 
{
    m_pData = new PASFAudioData();
    if ( ! (m_pData) ) return;
    
    // open the sound file and get info on desired channels and sample rate
    if (! (m_pData->sndfile = sf_open (m_pSalSound->m_aSoundFile.GetBuffer(), SFM_READ, &(m_pData->sfinfo)))) 
    {	
        // fprintf(stderr,"Error opening sound file %s\n", m_aSoundFile.GetBuffer()); fflush(stderr);
        m_pData->sndfile = NULL;
        return;
    }
    
    // check for valid number of channels to prevent problems later
    if (m_pData->sfinfo.channels < 1 || m_pData->sfinfo.channels > 2) 
    {	
        // fprintf(stderr, "Error incorrect number of channels %d.\n", m_pData->sfinfo.channels) ; fflush(stderr);
        sf_close(m_pData->sndfile);
        m_pData->sndfile = NULL;
        return;
    }
    
    
    // store information on desired settings and input buffer format away
    m_pData->format = paFloat32;
    m_pData->bufsize = PASF_BUFFER_LEN;
    
    // Using the default output device for this
    m_pData->device = Pa_GetDefaultOutputDeviceID();
    
    // the size of a frame in bytes is the number of channels of data * the size of the 
    // underlying data type type being used to store data for each for each channel
    // (we are using paFloat32 as input to PaOpenDefaultStream )
    
    unsigned int bytesPerFrame = m_pData->sfinfo.channels * sizeof(float);
    unsigned int framesPerBuffer = (unsigned int) (m_pData->bufsize / bytesPerFrame); 
    
    // Open an audio I/O stream for output only on the default output device
    PaError err = Pa_OpenDefaultStream(
    &m_pStream,
    0,                             // no input channels == no input device
    m_pData->sfinfo.channels,      // get desired output channels from file
    m_pData->format,               // 32 bit floating point format for each item
    m_pData->sfinfo.samplerate,    // get desired sampling rate from file
    framesPerBuffer,               // frames per buffer
    0,                             // number of buffers, if zero then use optimal number as determined by Pa
    pasf_in_out_callback,          // call back to load data into the stream
    m_pData );                     // pass along m_pData pointer so that callback can get access to info
    
    if( err != paNoError ) {
        m_pStream = NULL;
        sf_close(m_pData->sndfile);
        m_pData->sndfile = NULL;
    } 
}


PASFSound::~PASFSound()
{
    // if stream exists, stop and close it
    if (m_pStream) 
    { 
	    Pa_StopStream( m_pStream );
	    Pa_CloseStream( m_pStream );
        m_pStream = NULL;
    }
    // clean up the PASFAudioData structure, close any open sound files
    if ( m_pData ) 
    {
	    if (m_pData->sndfile) 
        {
            sf_close(m_pData->sndfile);
            m_pData->sndfile = NULL;
	    }
        delete m_pData;
	    m_pData = NULL;
	}
    m_pSalSound = NULL;
}


void PASFSound::play()
{
    int nErr = 1;
    BOOL bLoop = m_pSalSound->m_bLoop;
    ULONG nStartTime = m_pSalSound->m_nStartTime;
    ULONG nPlayTime =  m_pSalSound->m_nPlayTime;
    
    // if ready to go
    if ((m_pData) && (m_pStream)) 
    { 
        // set the player controls
        
        // if looping set the loop count (use -1 for infinity)
        m_pData->loop_count = 1;
        if ( bLoop ) m_pData->loop_count = -1;
        
        // convert StartTime to frame count and seek to starting frame in sound file
        m_pData->start_frame = 0;
        if (nStartTime > 0) 
        {
            m_pData->start_frame = (unsigned long)( (double)nStartTime * m_pData->sfinfo.samplerate / 1000.0 );
            sf_seek(m_pData->sndfile, m_pData->start_frame, SEEK_SET);
        }
        
        // set the frames number to play until (use 0 to indicate play until the end)
        m_pData->stop_frame = 0;
        if (nPlayTime != SOUND_PLAYALL) 
        {
            ULONG nEndTime = nStartTime + nPlayTime;
            m_pData->stop_frame = (unsigned long)( (double)nEndTime * m_pData->sfinfo.samplerate / 1000.0 );
        }
        
        // set set the total frame_count to the starting frame number
        m_pData->frame_count = m_pData->start_frame;
        m_pData->done_playing = 0;
        
        // FIXME?: In Play but already playing will be ignored (no error) is that right?
        nErr = 0;
        if (!(m_pSalSound->m_bPlaying)) 
        {            
            PaError err = Pa_StartStream( m_pStream );
            if( err != paNoError)  nErr = 1;
        }  
    }
    
    if (!nErr) 
    {
        m_pSalSound->changeStatePlay();
        return;
	}
    m_pSalSound->setError(SOUNDERR_GENERAL_ERROR);
}



void PASFSound::stop()
{
    int nErr = 1;
	if ( m_pStream ) 
    { 
        // stop and close the stream
        PaError err = Pa_StopStream( m_pStream );
        PaError err1 = Pa_CloseStream( m_pStream );
        if( (err == paNoError) && (err1 == paNoError) ) nErr = 0;
        m_pStream = NULL;
    }    
    
    // close the sound file sound since another file can be loaded after stop 
    if ((m_pData) && (!(m_pStream))) {
	    if (m_pData->sndfile) 
        {
            sf_close(m_pData->sndfile);
            m_pData->sndfile = NULL;
        }
    }
    
    if (!nErr) 
    {
        m_pSalSound->changeStateStop();
        return;
	}
    m_pSalSound->setError(SOUNDERR_GENERAL_ERROR);
    
}



void PASFSound::pause()
{
    int nErr = 1;
	if (m_pStream ) {
        PaError err = Pa_StopStream( m_pStream );
        if( err == paNoError ) nErr = 0;
	}
    
    if (!nErr) 
    {
        m_pSalSound->changeStatePause();
        return;
	}
    m_pSalSound->setError(SOUNDERR_GENERAL_ERROR);
}



void PASFSound::cont()
{
    int nErr = 1;
    if (m_pSalSound->m_bPaused) { 
        if (m_pStream) {
            PaError err = Pa_StartStream( m_pStream );
            if( err == paNoError ) nErr = 0;
        }
    }    
    
    if (!nErr) 
    {
        m_pSalSound->changeStateCont();
        return;
	}
    m_pSalSound->setError(SOUNDERR_GENERAL_ERROR);
}



BOOL PASFSound::isValid()
{
    if ((m_pData) && (m_pStream)) return TRUE;
    return FALSE;
}



#endif

