/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: drawinglayeranimation.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2006/02/09 14:47:51 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "basegfx/polygon/b2dpolygontools.hxx"
#include "canvas/debug.hxx"
#include "canvas/elapsedtime.hxx"
#include "activity.hxx"
#include "wakeupevent.hxx"
#include "drawshape.hxx"
#include "shapesubset.hxx"
#include "shapeattributelayerholder.hxx"
#include "slideshowcontext.hxx"
#include "rtl/math.hxx"
#include "vcl/metric.hxx"
#include "vcl/salbtype.hxx"
#include "vcl/canvastools.hxx"
#include "vcl/metaact.hxx"
#include "com/sun/star/beans/XPropertySet.hpp"
#include "com/sun/star/drawing/TextAnimationKind.hpp"
#include "com/sun/star/drawing/TextAnimationDirection.hpp"
#include "com/sun/star/drawing/TextHorizontalAdjust.hpp"
#include "com/sun/star/drawing/TextVerticalAdjust.hpp"
#include "com/sun/star/drawing/HomogenMatrix3.hpp"
#include "com/sun/star/awt/Rectangle.hpp"
#include "tools.hxx"
#include "boost/weak_ptr.hpp"
#include "boost/noncopyable.hpp"
#include <vector>

using namespace com::sun::star;
using namespace com::sun::star::uno;
using namespace ::presentation::internal;

namespace {

class ScrollTextAnimNode
{
    sal_uInt32                      mnDuration; // single duration
    sal_uInt32                      mnRepeat; // 0 -> endless
    double                          mfStart;
    double                          mfStop;
    sal_uInt32                      mnFrequency; // in ms
    // forth and back change at mnRepeat%2:
    bool                            mbAlternate;
    
public:
    ScrollTextAnimNode(
        sal_uInt32 nDuration, sal_uInt32 nRepeat, double fStart, double fStop,
        sal_uInt32 nFrequency, bool bAlternate)
        :   mnDuration(nDuration),
            mnRepeat(nRepeat),
            mfStart(fStart),
            mfStop(fStop),
            mnFrequency(nFrequency),
            mbAlternate(bAlternate)
        {}
    
    sal_uInt32 GetDuration() const { return mnDuration; }
    sal_uInt32 GetRepeat() const { return mnRepeat; }
    sal_uInt32 GetFullTime() const { return mnDuration * mnRepeat; }
    double GetStart() const { return mfStart; }
    double GetStop() const { return mfStop; }
    sal_uInt32 GetFrequency() const { return mnFrequency; }
    bool DoAlternate() const { return mbAlternate; }
    
    double GetStateAtRelativeTime(sal_uInt32 nRelativeTime) const;
};

double ScrollTextAnimNode::GetStateAtRelativeTime(
    sal_uInt32 nRelativeTime) const
{
    if(mnRepeat) {
        // ending
        const sal_uInt32 nRepeatCount(nRelativeTime / mnDuration);
        sal_uInt32 nFrameTime(nRelativeTime - (nRepeatCount * mnDuration));
        
        if(DoAlternate() && (nRepeatCount + 1L) % 2L) {
            nFrameTime = mnDuration - nFrameTime;
        }
        
        return mfStart + ((mfStop - mfStart) *
                          ((double)nFrameTime / (double)mnDuration));
    }
    else {
        // endless
        sal_uInt32 nFrameTime(nRelativeTime % mnDuration);
        
        if(DoAlternate()) {
            const sal_uInt32 nRepeatCount(nRelativeTime / mnDuration);
            
            if((nRepeatCount + 1L) % 2L) {
                nFrameTime = mnDuration - nFrameTime;
            }
        }
        
        return mfStart + ((mfStop - mfStart) * ((double)nFrameTime / (double)mnDuration));
    }
}

class ActivityImpl : public Activity,
                     private boost::noncopyable
{
public:
    virtual ~ActivityImpl();
    
    ActivityImpl(
        SlideShowContext const& rContext,
        boost::shared_ptr<WakeupEvent> const& pWakeupEvent,
        boost::shared_ptr<DrawShape> const& pDrawShape );
    
protected:
    // Disposable:
    virtual void dispose();
    // Activity:
    virtual double calcTimeLag() const;
    virtual bool perform();
    virtual bool isActive() const;
    virtual bool needsScreenUpdate() const;
    virtual void dequeued();
    virtual void end();
    
    void updateShapeAttributes( double fTime,
                                basegfx::B2DRectangle const& parentBounds );
    
    // Access to VisibleWhenSTarted flags
    sal_Bool IsVisibleWhenStarted() const { return mbVisisbleWhenStarted; }
    sal_Bool IsVisibleWhenStopped() const { return mbVisisbleWhenStopped; }
    
    // scroll horizontal? if sal_False, scroll is vertical.
    bool ScrollHorizontal() const {
        return (drawing::TextAnimationDirection_LEFT == meDirection ||
                drawing::TextAnimationDirection_RIGHT == meDirection);
    }
    
    // Access to StepWidth in logical units
    sal_uInt32 GetStepWidthLogic() const;
    
    // is the animation direction opposite?
    bool DoScrollForward() const {
        return (drawing::TextAnimationDirection_RIGHT == meDirection ||
                drawing::TextAnimationDirection_DOWN == meDirection);
    }

    // do alternate text directions?
    bool DoAlternate() const { return mbAlternate; }
    
    // do scroll in?
    bool DoScrollIn() const { return mbScrollIn; }
    
private:
    // Scroll helper methods
    void ImpForceScrollTextAnimNodes();
    ScrollTextAnimNode* ImpGetScrollTextAnimNode(
        sal_uInt32 nTime, sal_uInt32& rRelativeTime );
    
    // calculate the MixerState value for given time
    double GetMixerState(sal_uInt32 nTime);
    
private:
    SlideShowContext                            maContext;
    boost::shared_ptr<WakeupEvent>              mpWakeupEvent;
    boost::weak_ptr<DrawShape>                  mpParentDrawShape;
    DrawShapeSharedPtr                          mpDrawShape;
    ShapeSubsetSharedPtr                        mpShapeSubset;
    ShapeAttributeLayerHolder                   maShapeAttrLayer;
    GDIMetaFileSharedPtr                        mpMetaFile;
    canvas::tools::ElapsedTime                  maTimer;
    double                                      mfRotationAngle;
    bool                                        mbFirstPerform;
    bool                                        mbIsDisposed;
    bool                                        mbIsActive;
    drawing::TextAnimationKind                  meAnimKind;
    basegfx::B2DTuple                           maAnchorTrans;
    basegfx::B2DTuple                           maDistances;
    
    // The blink frequency in ms
    sal_uInt32 mnFrequency;
    
    // The repeat count, init to 0L which means endless
    sal_uInt32 mnRepeat;
    
    // Flag to decide if text will be shown when animation has ended
    bool mbVisisbleWhenStopped;
    bool mbVisisbleWhenStarted;

    // Flag decides if TextScroll alternates. Default is sal_False.
    bool mbAlternate;
    
    // Flag to remember if this is a simple scrollin text
    bool mbScrollIn;
    
    // start time for this animation
    sal_uInt32 mnStartTime;
    
    // The AnimationDirection
    drawing::TextAnimationDirection meDirection;
    
    // Get width per Step. Negative means pixel, positive logical units
    sal_Int32 mnStepWidth;
    
    // The single anim steps
    std::vector< ScrollTextAnimNode > maVector;
    
    // the scroll rectangle
    Rectangle maScrollRectangleLogic;
    
    // the paint rectangle
    Rectangle maPaintRectangleLogic;
};

double ActivityImpl::GetMixerState( sal_uInt32 nTime )
{
    if (meAnimKind == drawing::TextAnimationKind_BLINK) {
        // from AInfoBlinkText:
        double fRetval(0.0);
        sal_Bool bDone(sal_False);
        const sal_uInt32 nLoopTime(2 * mnFrequency);
        
        if(mnRepeat) {
            const sal_uInt32 nEndTime(mnRepeat * nLoopTime);
            
            if(nTime >= nEndTime) {
                if(mbVisisbleWhenStopped) {
                    fRetval = 0.0;
                }
                else {
                    fRetval = 1.0;
                }
                
                bDone = sal_True;
            }
        }
        
        if(!bDone) {
            sal_uInt32 nTimeInLoop(nTime % nLoopTime);
            fRetval = (double)nTimeInLoop / (double)nLoopTime;
        }
        
        return fRetval;
    }
    else {
        // from AInfoScrollText:
        double fRetval(0.0);
        ImpForceScrollTextAnimNodes();
        
        if(!maVector.empty()) {
            sal_uInt32 nRelativeTime;
            ScrollTextAnimNode* pNode =
                ImpGetScrollTextAnimNode(nTime, nRelativeTime);
            
            if(pNode) {
                // use node
                fRetval = pNode->GetStateAtRelativeTime(nRelativeTime);
            }
            else {
					// end of animation, take last entry's end
                fRetval = maVector[maVector.size() - 1L].GetStop();
            }
        }
        
        return fRetval;
    }
}

// Access to StepWidth in logical units
sal_uInt32 ActivityImpl::GetStepWidthLogic() const
{
    sal_uInt32 const PIXEL_TO_LOGIC = 10;
    
    sal_uInt32 nRetval(0L);
    
    if(mnStepWidth < 0L) {
        // is in pixels, convert to logical units
        nRetval = (-mnStepWidth * PIXEL_TO_LOGIC);
    }
    else if(mnStepWidth > 0L) {
        // is in logical units
        nRetval = mnStepWidth;
    }
    
    if(0L == nRetval) {
        // step 1 pixel, use calculated value
        nRetval = PIXEL_TO_LOGIC;
    }
    
    return nRetval;
}

void ActivityImpl::ImpForceScrollTextAnimNodes()
{
    if(maVector.empty())
    {
        // prepare values
        sal_uInt32 nLoopTime;
        double fZeroLogic, fOneLogic, fInitLogic, fDistanceLogic;
        double fZeroLogicAlternate = 0.0, fOneLogicAlternate = 0.0;
        double fZeroRelative, fOneRelative, fInitRelative,fDistanceRelative;
        
        if(ScrollHorizontal())
        {
            if(DoAlternate())
            {
                if(maPaintRectangleLogic.GetWidth() >
                   maScrollRectangleLogic.GetWidth())
                {
                    fZeroLogicAlternate = maScrollRectangleLogic.Right() - maPaintRectangleLogic.GetWidth();
                    fOneLogicAlternate = maScrollRectangleLogic.Left();
                }
                else {
                    fZeroLogicAlternate = maScrollRectangleLogic.Left();
                    fOneLogicAlternate = maScrollRectangleLogic.Right() - maPaintRectangleLogic.GetWidth();
                }
            }
            
            fZeroLogic = maScrollRectangleLogic.Left() - maPaintRectangleLogic.GetWidth();
            fOneLogic = maScrollRectangleLogic.Right();
            fInitLogic = maPaintRectangleLogic.Left();
        }
        else {
            if(DoAlternate())
            {
                if(maPaintRectangleLogic.GetHeight() > maScrollRectangleLogic.GetHeight())
                {
                    fZeroLogicAlternate = maScrollRectangleLogic.Bottom() - maPaintRectangleLogic.GetHeight();
                    fOneLogicAlternate = maScrollRectangleLogic.Top();
                }
                else {
                    fZeroLogicAlternate = maScrollRectangleLogic.Top();
                    fOneLogicAlternate = maScrollRectangleLogic.Bottom() - maPaintRectangleLogic.GetHeight();
                }
            }
            
            fZeroLogic = maScrollRectangleLogic.Top() - maPaintRectangleLogic.GetHeight();
            fOneLogic = maScrollRectangleLogic.Bottom();
            fInitLogic = maPaintRectangleLogic.Top();
        }
        
        fDistanceLogic = fOneLogic - fZeroLogic;
        fInitRelative = (fInitLogic - fZeroLogic) / fDistanceLogic;
        
        if(DoAlternate()) {
            fZeroRelative =
                (fZeroLogicAlternate - fZeroLogic) / fDistanceLogic;
            fOneRelative =
                (fOneLogicAlternate - fZeroLogic) / fDistanceLogic;
            fDistanceRelative = fOneRelative - fZeroRelative;
        }
        else {
            fZeroRelative = 0.0;
            fOneRelative = 1.0;
            fDistanceRelative = 1.0;
        }
        
        if(mnStartTime) {
            // Start time loop
            ScrollTextAnimNode aStartNode(
                mnStartTime, 1L, 0.0, 0.0, mnStartTime, false);
            maVector.push_back(aStartNode);
        }
        
        if(IsVisibleWhenStarted()) {
            double fRelativeStartValue, fRelativeEndValue,fRelativeDistance;
            
            if(DoScrollForward()) {
                fRelativeStartValue = fInitRelative;
                fRelativeEndValue = fOneRelative;
                fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
            }
            else {
                fRelativeStartValue = fInitRelative;
                fRelativeEndValue = fZeroRelative;
                fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
            }
            
            const double fNumberSteps =
                (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic();
            nLoopTime = FRound(fNumberSteps * mnFrequency);
            
            // init loop
            ScrollTextAnimNode aInitNode(
                nLoopTime, 1L,
                fRelativeStartValue, fRelativeEndValue,
                mnFrequency, false);
            maVector.push_back(aInitNode);
        }
        
        // prepare main loop values
        {
            double fRelativeStartValue, fRelativeEndValue, fRelativeDistance;
            
            if(DoScrollForward()) {
                fRelativeStartValue = fZeroRelative;
                fRelativeEndValue = fOneRelative;
                fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
            }
            else {
                fRelativeStartValue = fOneRelative;
                fRelativeEndValue = fZeroRelative;
                fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
            }
            
            const double fNumberSteps =
                (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic();
            nLoopTime = FRound(fNumberSteps * mnFrequency);
            
            if(0L == mnRepeat) {
                if(!DoScrollIn()) {
                    // endless main loop
                    ScrollTextAnimNode aMainNode(
                        nLoopTime, 0L,
                        fRelativeStartValue, fRelativeEndValue,
                        mnFrequency, DoAlternate());
                    maVector.push_back(aMainNode);
                }
            }
            else {
                sal_uInt32 nNumRepeat(mnRepeat);
                
                if(DoAlternate() && (nNumRepeat + 1L) % 2L) {
                    nNumRepeat += 1L;
                }
                
                // ending main loop
                ScrollTextAnimNode aMainNode(
                    nLoopTime, nNumRepeat,
                    fRelativeStartValue, fRelativeEndValue,
                    mnFrequency, DoAlternate());
                maVector.push_back(aMainNode);
            }
        }
        
        if(IsVisibleWhenStopped()) {
            double fRelativeStartValue, fRelativeEndValue, fRelativeDistance;
            
            if(DoScrollForward()) {
                fRelativeStartValue = fZeroRelative;
                fRelativeEndValue = fInitRelative;
                fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
            }
            else {
                fRelativeStartValue = fOneRelative;
                fRelativeEndValue = fInitRelative;
                fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
            }
            
            const double fNumberSteps =
                (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic();
            nLoopTime = FRound(fNumberSteps * mnFrequency);
            
            // exit loop
            ScrollTextAnimNode aExitNode(
                nLoopTime, 1L,
                fRelativeStartValue, fRelativeEndValue, mnFrequency, false);
            maVector.push_back(aExitNode);
        }
    }
}

ScrollTextAnimNode* ActivityImpl::ImpGetScrollTextAnimNode(
    sal_uInt32 nTime, sal_uInt32& rRelativeTime )
{
    ScrollTextAnimNode* pRetval = 0L;
    ImpForceScrollTextAnimNodes();
    
    if(!maVector.empty())
    {
        rRelativeTime = nTime;
        
        for(sal_uInt32 a(0L); !pRetval && a < maVector.size(); a++)
        {
            ScrollTextAnimNode & rNode = maVector[a];
            if(!rNode.GetRepeat()) {
                // endless loop, use it
                pRetval = &rNode;
            }
            else if(rNode.GetFullTime() > rRelativeTime) {
                // ending node
                pRetval = &rNode;
            }
            else {
                // look at next
                rRelativeTime -= rNode.GetFullTime();
            }
        }
    }
    
    return pRetval;
}

void ActivityImpl::updateShapeAttributes(
    double fTime, basegfx::B2DRectangle const& parentBounds )
{
    OSL_ASSERT( meAnimKind != drawing::TextAnimationKind_NONE );
    if (meAnimKind == drawing::TextAnimationKind_NONE)
        return;
    
    double const fMixerState = GetMixerState(
        static_cast<sal_uInt32>(fTime * 1000.0) );
    
    if (meAnimKind == drawing::TextAnimationKind_BLINK) {
        // show/hide text:
        maShapeAttrLayer.get()->setVisibility( fMixerState < 0.5 );
    }
    else if (mpMetaFile) { // scroll mode:
        
        //
        // keep care: the below code is highly sensible to changes...
        //
        
        // rectangle of the pure text:
        double const fPaintWidth = maPaintRectangleLogic.GetWidth();
        double const fPaintHeight = maPaintRectangleLogic.GetHeight();
        // rectangle where the scrolling takes place (-> clipping):
        double const fScrollWidth = maScrollRectangleLogic.GetWidth();
        double const fScrollHeight = maScrollRectangleLogic.GetHeight();
        
        basegfx::B2DPoint pos, clipPos, clipOffset;
        
        if (ScrollHorizontal()) {
            double const fOneEquiv( (fScrollWidth + fPaintWidth) / 2.0 );
            double const fZeroEquiv( -fOneEquiv );
            pos.setX( fZeroEquiv + (fMixerState * (fOneEquiv - fZeroEquiv))
                      - maAnchorTrans.getX() );
            
            // clip rect specified at upper left corner of _larger_ rectangle,
            // so check whether scroll rectangle still exceeds paint rectangle:
            if (fScrollWidth < fPaintWidth) {
                clipOffset.setX( (fPaintWidth - fScrollWidth) / 2.0
                                 - maAnchorTrans.getX() );
            }
            clipPos.setX( clipOffset.getX() - pos.getX() );
            clipPos.setY( -pos.getY() );
            pos.setY( pos.getY() + maAnchorTrans.getY() + maDistances.getY() );
        }
        else { // scroll vertical:
            double const fOneEquiv( (fScrollHeight + fPaintHeight) / 2.0 );
            double const fZeroEquiv( -fOneEquiv );
            pos.setY( fZeroEquiv + (fMixerState * (fOneEquiv - fZeroEquiv))
                      - maAnchorTrans.getY() );
            
            // clip rect specified at upper left corner of _larger_ rectangle,
            // so check whether scroll rectangle still exceeds paint rectangle:
            if (fScrollHeight < fPaintHeight) {
                clipOffset.setY( (fPaintHeight - fScrollHeight) / 2.0
                                 - maAnchorTrans.getY() );
            }
            clipPos.setX( -pos.getX() );
            clipPos.setY( clipOffset.getY() - pos.getY() );
            pos.setX( pos.getX() + maAnchorTrans.getX() + maDistances.getX() );
        }
        
        basegfx::B2DPolygon clipPoly(
            basegfx::tools::createPolygonFromRect(
                basegfx::B2DRectangle( clipPos.getX(),
                                       clipPos.getY(),
                                       clipPos.getX() + fScrollWidth,
                                       clipPos.getY() + fScrollHeight ) ) );
        
        if (!::basegfx::fTools::equalZero( mfRotationAngle )) {
            maShapeAttrLayer.get()->setRotationAngle( mfRotationAngle );
            double const fRotate = (mfRotationAngle * M_PI / 180.0);
            basegfx::B2DHomMatrix aTransform;
            // position:
            aTransform.rotate( fRotate );
            pos *= aTransform;
        }
        
        pos += parentBounds.getCenter();
        maShapeAttrLayer.get()->setPosition( pos );
        maShapeAttrLayer.get()->setClip( basegfx::B2DPolyPolygon(clipPoly) );
    }
}

bool ActivityImpl::perform()
{
    if (! isActive())
        return false;
    
    ENSURE_AND_RETURN(
        mpDrawShape.get() != 0,
        "ActivityImpl::perform(): still active, but NULL draw shape" );
    
    DrawShapeSharedPtr const pParentDrawShape( mpParentDrawShape );
    if (! pParentDrawShape) {
        // parent has vanished:
        return false;
    }
    
    if (pParentDrawShape->isVisible()) {
        if (mbFirstPerform) {
            mpDrawShape->setVisibility(true); // shape may be initially hidden
            maContext.mpLayerManager->enterAnimationMode( mpDrawShape );
            maTimer.reset();
            mbFirstPerform = false;
        }
        // update attributes related to current time:
        basegfx::B2DRectangle const parentBounds(
            pParentDrawShape->getPosSize() );
        updateShapeAttributes( maTimer.getElapsedTime(), parentBounds );
    }
    
    mpWakeupEvent->start();
    mpWakeupEvent->setNextTimeout( 0.1 );
    maContext.mrEventQueue.addEvent( mpWakeupEvent );
    
    if (mpDrawShape->isUpdateNecessary())
        maContext.mpLayerManager->notifyShapeUpdate( mpDrawShape );
    
    // don't reinsert, WakeupEvent will perform that after the given timeout:
    return false;
}

ActivityImpl::ActivityImpl(
    SlideShowContext const& rContext,
    boost::shared_ptr<WakeupEvent> const& pWakeupEvent,
    boost::shared_ptr<DrawShape> const& pParentDrawShape )
    : maContext(rContext),
      mpWakeupEvent(pWakeupEvent),
      mpParentDrawShape(pParentDrawShape),
      maTimer(rContext.mrEventQueue.getTimer()),
      mbFirstPerform(true),
      mbIsDisposed(false),
      mbIsActive(true),
      meAnimKind(drawing::TextAnimationKind_NONE),
      mnStartTime(0L)
{
    // get doctreenode:
    sal_Int32 const nNodes = pParentDrawShape->getNumberOfTreeNodes(
        DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH );

    DocTreeNode const startNode(
        pParentDrawShape->getTreeNode(
            0, DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH ) );
    if (nNodes > 1) {
        // xxx todo: remove this hack
        DocTreeNode const node(
            startNode.getStartIndex(),
            pParentDrawShape->getTreeNode(
                nNodes - 1,
                DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH ).getEndIndex(),
            DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH );
        mpShapeSubset.reset( new ShapeSubset( pParentDrawShape, node,
                                              rContext.mpLayerManager ) );
    }
    else {
        mpShapeSubset.reset( new ShapeSubset( pParentDrawShape, startNode,
                                              rContext.mpLayerManager ) );
    }
    mpShapeSubset->enableSubsetShape();
        
    mpDrawShape = boost::dynamic_pointer_cast<DrawShape>(
        mpShapeSubset->getSubsetShape() );
    mpMetaFile = mpDrawShape->forceScrollTextMetaFile();
    
    basegfx::B2DRectangle aScrollRect, aPaintRect;
    ENSURE_AND_THROW( getRectanglesFromScrollMtf( aScrollRect, 
                                                  aPaintRect,
                                                  mpMetaFile ),
                      "ActivityImpl::ActivityImpl(): Could not extract "
                      "scroll anim rectangles from mtf" );
    
    maScrollRectangleLogic = vcl::unotools::rectangleFromB2DRectangle(
        aScrollRect );
    maPaintRectangleLogic = vcl::unotools::rectangleFromB2DRectangle(
        aPaintRect );
    
    maShapeAttrLayer.createAttributeLayer(
        mpShapeSubset->getSubsetShape() );

    Reference<drawing::XShape> const xShape( mpDrawShape->getXShape() );
    Reference<beans::XPropertySet> const xProps( xShape, UNO_QUERY_THROW );
    
    //
    getPropertyValue( meAnimKind, xProps, OUSTR("TextAnimationKind") );
    OSL_ASSERT( meAnimKind != drawing::TextAnimationKind_NONE );    
    mbAlternate = (meAnimKind == drawing::TextAnimationKind_ALTERNATE);
    mbScrollIn = (meAnimKind == drawing::TextAnimationKind_SLIDE);
    
    // prepare text anchoring:
    drawing::TextVerticalAdjust eVerticalAdjust =
        drawing::TextVerticalAdjust_CENTER;
    drawing::TextHorizontalAdjust eHorizontalAdjust =
        drawing::TextHorizontalAdjust_CENTER;
    getPropertyValue( eVerticalAdjust, xProps, OUSTR("TextVerticalAdjust") );
    getPropertyValue( eHorizontalAdjust, xProps, OUSTR("TextHorizontalAdjust"));
    
    drawing::HomogenMatrix3 aTransformation;
    if ((eVerticalAdjust != drawing::TextVerticalAdjust_CENTER ||
        eHorizontalAdjust != drawing::TextHorizontalAdjust_CENTER) &&
        getPropertyValue( aTransformation, xProps, OUSTR("Transformation") ))
    {
        basegfx::B2DHomMatrix matrix;
        matrix.set( 0, 0, aTransformation.Line1.Column1 );
        matrix.set( 0, 1, aTransformation.Line1.Column2 );
        matrix.set( 0, 2, aTransformation.Line1.Column3 );
        matrix.set( 1, 0, aTransformation.Line2.Column1 );
        matrix.set( 1, 1, aTransformation.Line2.Column2 );
        matrix.set( 1, 2, aTransformation.Line2.Column3 );
        matrix.set( 2, 0, aTransformation.Line3.Column1 );
        matrix.set( 2, 1, aTransformation.Line3.Column2 );
        matrix.set( 2, 2, aTransformation.Line3.Column3 );
        basegfx::B2DTuple plainSize, translate;
        double rot, shear;
		if (matrix.decompose( plainSize /* scale */, translate, rot, shear )) {
            double fUpperDistance = 0.0, fLowerDistance = 0.0;
            double fLeftDistance = 0.0, fRightDistance = 0.0;
            // distances:
            getPropertyValue( fUpperDistance, xProps,
                              OUSTR("TextUpperDistance") );
            getPropertyValue( fLowerDistance, xProps,
                              OUSTR("TextLowerDistance") );
            getPropertyValue( fLeftDistance, xProps,
                              OUSTR("TextLeftDistance") );
            getPropertyValue( fRightDistance, xProps,
                              OUSTR("TextRightDistance") );
            switch (eVerticalAdjust) {
            case drawing::TextVerticalAdjust_TOP:
                maAnchorTrans.setY( -(plainSize.getY()
                                      -maPaintRectangleLogic.GetHeight()) /2.0);
                maDistances.setY( fUpperDistance );
                break;
            case drawing::TextVerticalAdjust_BOTTOM:
                maAnchorTrans.setY( (plainSize.getY()
                                     -maPaintRectangleLogic.GetHeight()) /2.0 );
                maDistances.setY( -fLowerDistance );
            default:
                break;
            }
            switch (eHorizontalAdjust) {
            case drawing::TextHorizontalAdjust_LEFT:
                maAnchorTrans.setX( -(plainSize.getX()
                                      -maPaintRectangleLogic.GetWidth()) /2.0 );
                maDistances.setX( fLeftDistance );
                break;
            case drawing::TextHorizontalAdjust_RIGHT:
                maAnchorTrans.setX( (plainSize.getX()
                                     -maPaintRectangleLogic.GetWidth()) / 2.0 );
                maDistances.setX( -fRightDistance );
            default:
                break;
            }
        }
        else
            OSL_ASSERT(false);
    }
    
    // adopted from in AInfoBlinkText::ImplInit():
    sal_Int16 nRepeat;
    getPropertyValue( nRepeat, xProps, OUSTR("TextAnimationCount") );
    mnRepeat = nRepeat;
    
    if (mbAlternate) {
        // force visible when started for scroll-forth-and-back, because
        // slide has been coming in with visible text in the middle:
        mbVisisbleWhenStarted = true;
    }
    else {
        getPropertyValue( mbVisisbleWhenStarted, xProps,
                          OUSTR("TextAnimationStartInside") );
    }
    
    // set visible when stopped
    getPropertyValue( mbVisisbleWhenStopped, xProps,
                      OUSTR("TextAnimatiogonStopInside") );
    // rotation:
    getPropertyValue( mfRotationAngle, xProps,
                      OUSTR("RotateAngle") );
    mfRotationAngle /= -100.0; // (switching direction)
    
    // set frequency
    sal_Int16 nDelay;
    getPropertyValue( nDelay, xProps, OUSTR("TextAnimationDelay") );
    // set delay if not automatic
    mnFrequency = (nDelay ? nDelay :
                   // default:
                   meAnimKind == drawing::TextAnimationKind_BLINK
                   ? 250L : 50L );
    
    // adopted from in AInfoScrollText::ImplInit():
    
    // If it is a simple m_bScrollIn, reset some parameters
    if (DoScrollIn()) {
        // most parameters are set correctly from the dialog logic, but
        // eg VisisbleWhenStopped is grayed out and needs to be corrected here.
        mbVisisbleWhenStopped = true;
        mbVisisbleWhenStarted = false;
        mnRepeat = 0L;
    }
    
    // Get animation direction
    getPropertyValue( meDirection, xProps, OUSTR("TextAnimationDirection") );
    
    // Get step width. Negative means pixel, positive logical units
    getPropertyValue( mnStepWidth, xProps, OUSTR("TextAnimationAmount") );

    // make text invisible for slide transition bitmaps
    // if !TextAnimationStartInside
    // else start in the middle, where the shape text
    // has been rendered before...
    if (! mbVisisbleWhenStarted) {
        mpDrawShape->setVisibility(false);
    }
    
    // update position for start:
    basegfx::B2DRectangle const parentBounds( pParentDrawShape->getPosSize() );
    updateShapeAttributes( 0.0, parentBounds );
}

ActivityImpl::~ActivityImpl()
{
}

void ActivityImpl::dispose()
{
    if (! mbIsDisposed) {
        mbIsActive = false;
        maContext.mpLayerManager->leaveAnimationMode( mpDrawShape );
        maShapeAttrLayer.reset();
        if (mpShapeSubset.get() != 0) {
            mpShapeSubset->disableSubsetShape();
            mpShapeSubset.reset();
        }
        mpMetaFile.reset();
        mpDrawShape.reset();
        mpParentDrawShape.reset();
        mpWakeupEvent.reset();
        maContext.dispose();
        mbIsDisposed = true;
    }
}

double ActivityImpl::calcTimeLag() const
{
    return 0.0;
}

bool ActivityImpl::isActive() const
{
    return mbIsActive;
}

bool ActivityImpl::needsScreenUpdate() const
{
    return (isActive() && mpDrawShape->isUpdateNecessary());
}

void ActivityImpl::dequeued()
{
    // not used here
}

void ActivityImpl::end()
{
    mbIsActive = false;
    
    DrawShapeSharedPtr const pParentDrawShape( mpParentDrawShape );
    if (! pParentDrawShape) {
        // parent has vanished:
        return;
    }
    
    if (pParentDrawShape->isVisible()) {
        if (mbFirstPerform) {
            mpDrawShape->setVisibility(true); // shape may be initially hidden
            maContext.mpLayerManager->enterAnimationMode( mpDrawShape );
            maTimer.reset();
            mbFirstPerform = false;
        }
        
        // update attributes related to current time:
        basegfx::B2DRectangle const parentBounds(
            pParentDrawShape->getPosSize() );
        updateShapeAttributes( maTimer.getElapsedTime(), parentBounds );
        
        if (mpDrawShape->isUpdateNecessary())
            maContext.mpLayerManager->notifyShapeUpdate( mpDrawShape );
    }
}

} // anon namespace

namespace presentation {
namespace internal {

boost::shared_ptr<Activity> createDrawingLayerAnimActivity(
    SlideShowContext const& rContext,
    boost::shared_ptr<DrawShape> const& pDrawShape )
{
    boost::shared_ptr<WakeupEvent> const pWakeupEvent(
        new WakeupEvent( rContext.mrEventQueue.getTimer(),
                         rContext.mrActivitiesQueue ) );
    boost::shared_ptr<Activity> pActivity(
        new ActivityImpl( rContext, pWakeupEvent, pDrawShape ) );
    pWakeupEvent->setActivity( pActivity );
    return pActivity;
}

} // namespace internal
} // namespace presentation

