/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: AccessibleBase.cxx,v $
 *
 *  $Revision: 1.23 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 00:00:21 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include "AccessibleBase.hxx"

#ifndef _COM_SUN_STAR_ACCESSIBILITY_ACCESSIBLEROLE_HPP_
#include <com/sun/star/accessibility/AccessibleRole.hpp>
#endif
#ifndef _COM_SUN_STAR_ACCESSIBILITY_ACCESSIBLEEVENTID_HPP_
#include <com/sun/star/accessibility/AccessibleEventId.hpp>
#endif
#ifndef _COM_SUN_STAR_ACCESSIBILITY_ACCESSIBLESTATETYPE_HPP_
#include <com/sun/star/accessibility/AccessibleStateType.hpp>
#endif

#ifndef _RTL_USTRBUF_HXX_ 
#include <rtl/ustrbuf.hxx>
#endif
#ifndef _SV_SVAPP_HXX
#include <vcl/svapp.hxx>
#endif
#ifndef SVX_UNOPROV_HXX 
#include <svx/unoprov.hxx>
#endif
#ifndef _SFXHINT_HXX
#include <svtools/hint.hxx>
#endif
#ifndef _VCL_UNOHELP_HXX
#include <vcl/unohelp.hxx>
#endif

// STL
#ifndef _STLP_ALGORITHM
#include <algorithm>
#endif

#ifndef _CHTMODEL_HXX
#include "chtmodel.hxx"
#endif

#include "ChartElementFactory.hxx"

using namespace ::com::sun::star;
using namespace ::com::sun::star::accessibility;

using ::com::sun::star::uno::UNO_QUERY;
using ::rtl::OUString;
using ::rtl::OUStringBuffer;
using ::com::sun::star::uno::Reference;
using ::osl::MutexGuard;
using ::osl::ClearableMutexGuard;
using ::osl::ResettableMutexGuard;
using ::com::sun::star::uno::RuntimeException;
using ::com::sun::star::uno::Any;

namespace accessibility
{

/** @param bMayHaveChildren is false per default
 */
AccessibleBase::AccessibleBase(
    const AccessibleUniqueId & rId,
    AccessibleBase * pParent,
    bool bMayHaveChildren ) :

        m_bIsDisposed( false ),
        m_aDisposeListenerList( m_aAccBaseMutex ),
        m_bMayHaveChildren( bMayHaveChildren ),
        m_bChildrenInitialized( false ),
        m_pParent( pParent ),
        m_pModel( NULL ),
        m_pWindow( NULL ),
        m_aId( rId ),
        m_pStateSetHelper( new ::utl::AccessibleStateSetHelper() ),
        m_aStateSet( m_pStateSetHelper ),
        m_nEventNotifierId(0)
{
    if( m_pParent != NULL )
    {
        m_pModel =  m_pParent->GetChartModel();
        m_pWindow = m_pParent->GetWindow();
    }

    // initialize some states
    OSL_ASSERT( m_pStateSetHelper );
    m_pStateSetHelper->AddState( AccessibleStateType::ENABLED );
    m_pStateSetHelper->AddState( AccessibleStateType::SHOWING );
    m_pStateSetHelper->AddState( AccessibleStateType::VISIBLE );
    m_pStateSetHelper->AddState( AccessibleStateType::SELECTABLE );
    m_pStateSetHelper->AddState( AccessibleStateType::FOCUSABLE );
}

AccessibleBase::~AccessibleBase()
{
    OSL_ASSERT( m_bIsDisposed );
}

// ________ public ________

bool AccessibleBase::CheckDisposeState( bool bThrowException /* default: true */ ) const
    throw (lang::DisposedException)
{
    if( bThrowException &&
        m_bIsDisposed )
    {
        throw lang::DisposedException(
            OUString( RTL_CONSTASCII_USTRINGPARAM( "component has state DEFUNC" )),
            static_cast< uno::XWeak * >( const_cast< AccessibleBase * >( this ) ) );
    }

    
    return m_bIsDisposed;
}

::osl::Mutex &  AccessibleBase::GetMutex() const
{
    return m_aAccBaseMutex;
}

// ________ protected ________

/** This method is not declared pure virtual for elements which do not support
    having children such that overloading would be unnecessary
 */
bool AccessibleBase::UpdateChildren()
{
    OSL_ENSURE( ! m_bMayHaveChildren,
                "This method should be overloaded by derived class or never be called" );
    return false;
}

awt::Point AccessibleBase::GetUpperLeftOnScreen() const
{
    awt::Point aResult;
    if( m_pParent )
    {
        // /--
        ClearableMutexGuard aGuard( m_aAccBaseMutex );
        AccessibleBase * pParent = m_pParent;
        aGuard.clear();
        // \--

        if( pParent )
        {
            aResult = pParent->GetUpperLeftOnScreen();
        }
        else
            OSL_ENSURE( false, "Default position used is probably incorrect." );
    }

    return aResult;
}

void AccessibleBase::AddChild( AccessibleBase * pChild  )
{
    OSL_ENSURE( pChild != NULL, "Invalid Child" );
    if( pChild )
    {
        // /--
        ClearableMutexGuard aGuard( m_aAccBaseMutex );

        Reference< XAccessible > xChild( pChild );
        m_aChildList.push_back( xChild );
        m_aChildIndexAccess[ pChild->GetId() ] = ( m_aChildList.size() - 1 );

        // inform listeners of new child
        if( m_bChildrenInitialized )
        {
            Any aEmpty, aNew;
            aNew <<= xChild;

            aGuard.clear();
            // \-- (1st)
            BroadcastAccEvent( AccessibleEventId::CHILD, aNew, aEmpty );
        }
        // \-- (2nd)
    }
}

/** in this method we imply that the Reference< XAccessible > elements in the
    vector are AccessibleBase objects !
 */
void AccessibleBase::RemoveChildById( const AccessibleUniqueId & rId )
{
    Reference< XAccessible > aChild = GetChild( rId );

    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );

    // search child in vector
    ChildListVectorType::iterator aVecIter =
          ::std::find( m_aChildList.begin(), m_aChildList.end(), aChild );
    OSL_ENSURE( aVecIter != m_aChildList.end(),
                "trying to erase non-available child from vector" );

    aGuard.clear();
    // \--

    RemoveChild( aVecIter );
}

sal_Int32 AccessibleBase::GetChildIndex( const Reference< XAccessible > & rChild ) const
{
    bool bFound = false;
    sal_Int32 nIndex = 0;

    // /--
    MutexGuard aGuard( m_aAccBaseMutex );

    for( ChildListVectorType::const_iterator aIter( m_aChildList.begin() );
         aIter != m_aChildList.end();
         ++aIter, ++nIndex )
    {
        if( *aIter == rChild )
        {
            bFound = true;
            break;
        }
    }

    return (bFound ? nIndex : static_cast< sal_Int32 >( -1 ));
    // \--
}

bool AccessibleBase::UpdateChild(
    const AccessibleUniqueId & rId,
    bool bOldState,
    bool bNewState )
{
    if( bOldState != bNewState )
    {
        if( bNewState )
            AddChild( ChartElementFactory::CreateChartElement( this, rId ));
        else
        {
            RemoveChildById( rId );
        }
    }    

    return bNewState;
}

ChartModel * AccessibleBase::GetChartModel() const
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    return m_pModel;
    // \--
}

void AccessibleBase::SetChartModel( ChartModel * pModel )
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    OSL_ENSURE( m_pModel == NULL, "trying to set a new model" );
    m_pModel = pModel;
    // \--
}

SchWindow * AccessibleBase::GetWindow() const
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    return m_pWindow;
    // \--
}

void AccessibleBase::SetWindow( SchWindow * pWindow )
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    OSL_ENSURE( m_pWindow == NULL, "trying to set a new window" );
    m_pWindow = pWindow;
    // \--
}

const AccessibleUniqueId & AccessibleBase::GetId() const
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    return m_aId;
    // \--
}

Reference< XAccessible > AccessibleBase::GetChild( const AccessibleUniqueId & rId ) const
{
    Reference< XAccessible > aResult;

    // /--
    MutexGuard aGuard( m_aAccBaseMutex );

    ChildIndexHashType::const_iterator aIter = m_aChildIndexAccess.find( rId );
    if( aIter != m_aChildIndexAccess.end() )
    {
        // found a valid index in the hash
        aResult.set( m_aChildList[ (*aIter).second ] );
    }

    return aResult;
    // \--
}

bool AccessibleBase::HasChild( const AccessibleUniqueId & rId ) const
{
    // /--
    MutexGuard aGuard( m_aAccBaseMutex );
    ChildIndexHashType::const_iterator aIter = m_aChildIndexAccess.find( rId );
    return ( aIter != m_aChildIndexAccess.end() );
    // \--
}

bool AccessibleBase::NotifyEvent( EventType eEventId,
                                  const AccessibleUniqueId & rId,
                                  const SfxHint * pHint /* default: 0 */  )
{
    if( m_bMayHaveChildren )
    {
        bool bStop = false;
        // /--
        ClearableMutexGuard aGuard( m_aAccBaseMutex );
        // make local copy for notification
        ChildListVectorType aLocalChildList( m_aChildList );
        aGuard.clear();
        // \--

        ChildListVectorType::iterator aEndIter = aLocalChildList.end();
        for( ChildListVectorType::iterator aIter = aLocalChildList.begin() ;
             ( aIter != aEndIter ) && ( ! bStop ) ;
             ++aIter )
        {
            // Note: at this place we must be sure to have an AccessibleBase
            // object in the UNO reference to XAccessible !
            bStop = (*static_cast< AccessibleBase * >
                     ( (*aIter).get() )).NotifyEvent( eEventId, rId, pHint );
        }
        return bStop;
    }

    return false;
}

void AccessibleBase::AddState( sal_Int16 aState )
    throw (RuntimeException)
{
    CheckDisposeState();
    OSL_ASSERT( m_pStateSetHelper );
    m_pStateSetHelper->AddState( aState );
}

void AccessibleBase::RemoveState( sal_Int16 aState )
    throw (RuntimeException)
{
    CheckDisposeState();
    OSL_ASSERT( m_pStateSetHelper );
    m_pStateSetHelper->RemoveState( aState );
}

/// type that must be supported by selection listeners
inline const ::com::sun::star::uno::Type & lcl_getAccEventListenerType()
{
	return ::getCppuType( (Reference< XAccessibleEventListener > *)0 );
}

void AccessibleBase::BroadcastAccEvent(
    sal_Int16 nId,
    const Any & rNew,
    const Any & rOld,
    bool bSendGlobally ) const
{
    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );

    if ( !m_nEventNotifierId && !bSendGlobally )
        return;
		// if we don't have a client id for the notifier, then we don't have listeners, then
		// we don't need to notify anything
        //except SendGlobally for focus handling?

    // the const cast is needed, because UNO parameters are never const
    const AccessibleEventObject aEvent(
        const_cast< uno::XWeak * >( static_cast< const uno::XWeak * >( this )),
        nId, rNew, rOld );

    if ( m_nEventNotifierId ) // let the notifier handle this event
		::comphelper::AccessibleEventNotifier::addEvent( m_nEventNotifierId, aEvent );

    aGuard.clear();
    // \--

    // send event to global message queue
    if( bSendGlobally )
    {
        ::vcl::unohelper::NotifyAccessibleStateEventGlobally( aEvent );
    }
}

void AccessibleBase::KillAllChildren()
{
    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );

    // make local copy for notification
    ChildListVectorType aLocalChildList( m_aChildList );

    // remove all children
    m_aChildList.clear();
    m_aChildIndexAccess.clear();

    aGuard.clear();
    // \--

    // call dispose for all children
    // and notify listeners
    Reference< lang::XComponent > xComp;
    Any aEmpty, aOld;
    ChildListVectorType::const_iterator aEndIter = aLocalChildList.end();
    for( ChildListVectorType::const_iterator aIter = aLocalChildList.begin();
         aIter != aEndIter; ++aIter )
    {
        aOld <<= (*aIter);
        BroadcastAccEvent( AccessibleEventId::CHILD, aEmpty, aOld );

        xComp.set( *aIter, UNO_QUERY );
        if( xComp.is())
            xComp->dispose();
    }
}

// ________ private ________

void AccessibleBase::RemoveChild( const ChildListVectorType::iterator& rIter )
{
    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );
    Reference< XAccessible > xChild;

    if( rIter != m_aChildList.end() )
    {
        xChild.set( *rIter );

        // remove child from vector
        m_aChildList.erase( rIter );

        // all indices may have changed => rebuild hash map
        m_aChildIndexAccess.clear();
        ::std::vector< Reference< XAccessible > >::size_type nIndex = 0;
        for( ChildListVectorType::const_iterator aVecIter = m_aChildList.begin();
             aVecIter != m_aChildList.end();
             ++aVecIter, ++nIndex )
        {
            // Note: at this place we must be sure to have an AccessibleBase
            // object in the UNO reference to XAccessible !
            const AccessibleBase * pImpl =
                static_cast< const AccessibleBase * >( (*aVecIter).get() );
            m_aChildIndexAccess[ pImpl->GetId() ] = nIndex;
        }

        bool bInitialized = m_bChildrenInitialized;

        // call listeners unguarded
        aGuard.clear();
        // \-- (1st)

        // inform listeners of removed child
        if( bInitialized )
        {
            Any aEmpty, aOld;
            aOld <<= xChild;

            BroadcastAccEvent( AccessibleEventId::CHILD, aEmpty, aOld );
        }

        // dispose the child
        Reference< lang::XComponent > xComp( xChild, UNO_QUERY );
        if( xComp.is())
            xComp->dispose();
    }
    // \-- (2nd)
}


// ____________________________________
// ____________________________________
//
//             Interfaces
// ____________________________________
// ____________________________________

// ________ AccessibleBase::XComponent ________
void SAL_CALL AccessibleBase::dispose()
    throw (RuntimeException)
{
    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );
    OSL_ENSURE( ! m_bIsDisposed, "dispose() called twice" );

    // notify disposing to all AccessibleEvent listeners asynchron
	if ( m_nEventNotifierId )
	{
		::comphelper::AccessibleEventNotifier::revokeClientNotifyDisposing( m_nEventNotifierId, *this );
		m_nEventNotifierId = 0;
	}

    // reset pointers
    m_pParent = NULL;
    m_pModel = NULL;
    m_pWindow = NULL;

    // invalidate implementation for helper, but keep UNO reference to still
    // allow a tool to query the DEFUNC state.
    // Note: The object will be deleted when the last reference is released
    m_pStateSetHelper = NULL;

    // attach new empty state set helper to member reference
    ::utl::AccessibleStateSetHelper * pHelper = new ::utl::AccessibleStateSetHelper();
    pHelper->AddState( AccessibleStateType::DEFUNC );
    // release old helper and attach new one
    m_aStateSet.set( pHelper );

    m_bIsDisposed = true;

    lang::EventObject aEvent( *this );

    // call listeners unguarded
    aGuard.clear();
    // \--

    if( m_bMayHaveChildren )
    {
        KillAllChildren();
    }
    else
        OSL_ENSURE( m_aChildList.empty(), "Child list should be empty" );

    // notify all event listeners that registered as such
	m_aDisposeListenerList.disposeAndClear( aEvent );
}

void SAL_CALL AccessibleBase::addEventListener( const Reference< lang::XEventListener >& xListener )
    throw (RuntimeException)
{
    if( m_bIsDisposed )
    {
        lang::EventObject aEvent( static_cast< uno::XWeak * >( this ));
        xListener->disposing( aEvent );
    }
    else
        m_aDisposeListenerList.addInterface( xListener );
}

void SAL_CALL AccessibleBase::removeEventListener( const Reference< lang::XEventListener >& aListener )
    throw (RuntimeException)
{
    CheckDisposeState();
    m_aDisposeListenerList.removeInterface( aListener );
}

// ________ XAccessible ________
Reference< XAccessibleContext > SAL_CALL AccessibleBase::getAccessibleContext()
    throw (RuntimeException)
{
    return this;
}

// ________ AccessibleBase::XAccessibleContext ________
sal_Int32 SAL_CALL AccessibleBase::getAccessibleChildCount() 
    throw (RuntimeException)
{
    // /--
    ClearableMutexGuard aGuard( m_aAccBaseMutex );
    if( ! m_bMayHaveChildren ||
        m_bIsDisposed )
        return 0;

    bool bMustUpdateChildren = ( m_bMayHaveChildren &&
                                 ! m_bChildrenInitialized );
    // prevent a second initialisation during unguarded call
//     if( bMustUpdateChildren )
//         m_bChildrenInitialized = true;

    aGuard.clear();
    // \--

    // update unguarded
    if( bMustUpdateChildren )
    {
        bool bResult = UpdateChildren();
        if( bResult )
//         if( ! bResult )
        {
            // initialisation failed

            // /--
            MutexGuard aGuardAgain( m_aAccBaseMutex );
//             m_bChildrenInitialized = false;
            m_bChildrenInitialized = true;
            // \--
        }
    }

    return m_aChildList.size();
}

Reference< XAccessible > SAL_CALL AccessibleBase::getAccessibleChild( sal_Int32 i ) 
    throw (lang::IndexOutOfBoundsException, RuntimeException)
{
    CheckDisposeState();
    Reference< XAccessible > aResult;

    // /--
    ResettableMutexGuard aGuard( m_aAccBaseMutex );
    bool bMustUpdateChildren = ( m_bMayHaveChildren &&
                                 ! m_bChildrenInitialized );
    // prevent a second initialisation during unguarded call
//     if( bMustUpdateChildren )
//         m_bChildrenInitialized = true;

    aGuard.clear();
    // \--    

    if( bMustUpdateChildren )
    {
        bool bResult = UpdateChildren();
//         if( ! bResult )
        if( bResult )
        {
            // initialisation failed

            // /--
            aGuard.reset();
//             m_bChildrenInitialized = false;
            m_bChildrenInitialized = true;
            aGuard.clear();
            // \--
        }
    }

    // /--
    aGuard.reset();
    if( ! m_bMayHaveChildren ||
        i < 0 ||
        static_cast< VectorIndexType >( i ) >= m_aChildList.size() )
    {
        OUStringBuffer aBuf;
        aBuf.appendAscii( RTL_CONSTASCII_STRINGPARAM( "Index " ));
        aBuf.append( i );
        aBuf.appendAscii( RTL_CONSTASCII_STRINGPARAM( " is invalid for range [ 0, " ));
        aBuf.append( static_cast< sal_Int32 >( m_aChildList.size() - 1 ) );
        aBuf.appendAscii( RTL_CONSTASCII_STRINGPARAM( " ]" ) );
        lang::IndexOutOfBoundsException aEx( aBuf.makeStringAndClear(), static_cast< ::cppu::OWeakObject * >( this ));
        throw aEx;
    }
    else
        aResult.set( m_aChildList[ i ] );

    return aResult;
    // \--
}

Reference< XAccessible > SAL_CALL AccessibleBase::getAccessibleParent() 
    throw (RuntimeException)
{
    CheckDisposeState();
    Reference< XAccessible > aResult;
    if( m_pParent )
        aResult.set( m_pParent );

    return aResult;
}

sal_Int32 SAL_CALL AccessibleBase::getAccessibleIndexInParent() 
    throw (RuntimeException)
{
    CheckDisposeState();
    AccessibleBase *  pParent = m_pParent;

    if( pParent != NULL )
    {
        return pParent->GetChildIndex( this );
    }

    return static_cast< sal_Int32 >( -1 );
}

sal_Int16 SAL_CALL AccessibleBase::getAccessibleRole() 
    throw (RuntimeException)
{
    return AccessibleRole::SHAPE;
}

OUString SAL_CALL AccessibleBase::getAccessibleName()
    throw (RuntimeException)
{
    OUString aResult( getImplementationName());
    if( m_pParent != NULL )
        aResult += OUString::valueOf( getAccessibleIndexInParent());
    return aResult;
}

Reference< XAccessibleRelationSet > SAL_CALL AccessibleBase::getAccessibleRelationSet() 
    throw (RuntimeException)
{
    Reference< XAccessibleRelationSet > aResult;
    return aResult;
}

Reference< XAccessibleStateSet > SAL_CALL AccessibleBase::getAccessibleStateSet()
    throw (RuntimeException)
{
    // returns a valid object, even if this is already disposed.  This way it is
    // possible to return a state-set containing the DEFUNC state
    return m_aStateSet;
}

lang::Locale SAL_CALL AccessibleBase::getLocale() 
    throw (IllegalAccessibleComponentStateException, RuntimeException)
{
    CheckDisposeState();

    return Application::GetSettings().GetLocale();
}

// ________ AccessibleBase::XAccessibleComponent ________
sal_Bool SAL_CALL AccessibleBase::containsPoint( const awt::Point& aPoint ) 
    throw (RuntimeException)
{
    awt::Rectangle aRect( getBounds() );

    // contains() works with relative coordinates
    aRect.X = 0;
    aRect.Y = 0;

    return ( aPoint.X >= aRect.X &&
             aPoint.Y >= aRect.Y &&
             aPoint.X < (aRect.X + aRect.Width) &&
             aPoint.Y < (aRect.Y + aRect.Height) );
}

Reference< XAccessible > SAL_CALL AccessibleBase::getAccessibleAtPoint( const awt::Point& aPoint ) 
    throw (RuntimeException)
{
    CheckDisposeState();
    Reference< XAccessible > aResult;
    awt::Rectangle aRect( getBounds());

    // children are positioned relative to this object, so translate bound rect
    aRect.X = 0;
    aRect.Y = 0;

    // children must be inside the own bound rect
    if( ( aRect.X <= aPoint.X && aPoint.X <= (aRect.X + aRect.Width) ) &&
        ( aRect.Y <= aPoint.Y && aPoint.Y <= (aRect.Y + aRect.Height)))
    {
        // /--
        ClearableMutexGuard aGuard( m_aAccBaseMutex );
        ChildListVectorType aLocalChildList( m_aChildList );
        aGuard.clear();
        // \--

        Reference< XAccessibleComponent > aComp;
        for( ChildListVectorType::const_iterator aIter = aLocalChildList.begin();
             aIter != aLocalChildList.end(); ++aIter )
        {
            aComp.set( *aIter, UNO_QUERY );
            if( aComp.is())
            {
                aRect = aComp->getBounds();
                if( ( aRect.X <= aPoint.X && aPoint.X <= (aRect.X + aRect.Width) ) &&
                    ( aRect.Y <= aPoint.Y && aPoint.Y <= (aRect.Y + aRect.Height)))
                {
                    aResult = (*aIter);
                    break;
                }
            }
        }
    }

    return aResult;
}

awt::Point SAL_CALL AccessibleBase::getLocation()
    throw (RuntimeException)
{
    CheckDisposeState();
    awt::Rectangle aBBox( getBounds() );
    return awt::Point( aBBox.X, aBBox.Y );
}

awt::Point SAL_CALL AccessibleBase::getLocationOnScreen() 
    throw (RuntimeException)
{
    CheckDisposeState();

    if( m_pParent != NULL )
    {
        AccessibleBase * pParent = m_pParent;
        awt::Point aLocThisRel( getLocation());
        awt::Point aUpperLeft;

        if( pParent != NULL )
            aUpperLeft = pParent->getLocationOnScreen();

        return  awt::Point( aUpperLeft.X + aLocThisRel.X,
                            aUpperLeft.Y + aLocThisRel.Y );
    }
    else
        return getLocation();
}

awt::Size SAL_CALL AccessibleBase::getSize()
    throw (RuntimeException)
{
    CheckDisposeState();
    awt::Rectangle aBBox( getBounds() );
    return awt::Size( aBBox.Width, aBBox.Height );
}

void SAL_CALL AccessibleBase::grabFocus() 
    throw (RuntimeException)
{
}

sal_Int32 SAL_CALL AccessibleBase::getForeground()
    throw (RuntimeException)
{
    // return anything (transparent) as default
    return static_cast< sal_Int32 >( Color( COL_TRANSPARENT ).GetColor());
}

sal_Int32 SAL_CALL AccessibleBase::getBackground()
    throw (RuntimeException)
{
    // return anything (transparent) as default
    return static_cast< sal_Int32 >( Color( COL_TRANSPARENT ).GetColor());
}


// ________ AccessibleBase::XServiceInfo ________
OUString SAL_CALL AccessibleBase::getImplementationName() 
    throw (RuntimeException)
{
    return OUString( RTL_CONSTASCII_USTRINGPARAM( "AccessibleBase" ));
}

sal_Bool SAL_CALL AccessibleBase::supportsService( const OUString& ServiceName ) 
    throw (RuntimeException)
{
	return SvxServiceInfoHelper::supportsService( ServiceName, getSupportedServiceNames() );
}

uno::Sequence< OUString > SAL_CALL AccessibleBase::getSupportedServiceNames() 
    throw (RuntimeException)
{
	uno::Sequence< ::rtl::OUString > aSeq( 2 );
	::rtl::OUString* pStr = aSeq.getArray();
	pStr[ 0 ] = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "com.sun.star.accessibility.Accessible" ));
	pStr[ 1 ] = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "com.sun.star.accessibility.AccessibleContext" ));

	return aSeq;
}

// ________ AccessibleBase::XEventListener ________
void SAL_CALL AccessibleBase::disposing( const lang::EventObject& Source ) 
    throw (RuntimeException)
{
}

// ________ XAccessibleEventBroadcasters ________
void SAL_CALL AccessibleBase::addEventListener( const Reference< XAccessibleEventListener >& xListener ) 
    throw (uno::RuntimeException)
{
    MutexGuard aGuard( m_aAccBaseMutex );

	if ( xListener.is() )
	{
		if ( !m_nEventNotifierId )
			m_nEventNotifierId = ::comphelper::AccessibleEventNotifier::registerClient();

		::comphelper::AccessibleEventNotifier::addEventListener( m_nEventNotifierId, xListener );
	}
}

void SAL_CALL AccessibleBase::removeEventListener( const Reference< XAccessibleEventListener >& xListener ) 
    throw (uno::RuntimeException)
{
    MutexGuard aGuard( m_aAccBaseMutex );

	if ( xListener.is() )
	{
		sal_Int32 nListenerCount = ::comphelper::AccessibleEventNotifier::removeEventListener( m_nEventNotifierId, xListener );
		if ( !nListenerCount )
		{
			// no listeners anymore
			::comphelper::AccessibleEventNotifier::revokeClient( m_nEventNotifierId );
			m_nEventNotifierId = 0;
		}
	}
}

}       // namespace accessibility
